package com.northpool.service.manager.data_service;

import com.northpool.exception.UException;
import com.northpool.resources.command.Constants;
import com.northpool.resources.command.QueryFilter;
import com.northpool.resources.datatable.ITable;
import com.northpool.resources.datatable.dao.IScroll;
import com.northpool.service.config.data_service.DataServiceBuilder;
import com.northpool.service.config.data_service.IDataService;
import com.northpool.service.config.style.IStyleService;
import com.northpool.service.config.style.StyleJson;
import com.northpool.service.config.style.layer.AbstractLayer;
import com.northpool.service.config.style.layer.Layer;
import com.northpool.service.config.vector_service.IVectorService;
import com.northpool.service.config.vector_service.VectorServiceBuilder;
import com.northpool.service.config.vector_service.layer.ILayer;
import com.northpool.service.config.vector_service.layer.ILayerLevel;
import com.northpool.service.config.vector_service.layer.LayerBean;
import com.northpool.service.config.vector_service.storage.IStorageInfo;
import com.northpool.service.dao.data_service.DataServiceMongoDao;
import com.northpool.service.dao.data_service.DataServiceZkDao;
import com.northpool.service.manager.abstractclass.*;
import com.northpool.service.manager.style.IStyleManager;
import com.northpool.service.manager.vector_service.IVectorServiceManager;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

public class DataServiceManager extends ServiceManager<IDataService, DataServiceBuilder> implements IDataServiceManager  {
	
	protected static final String MANAGER_ROOT = "data_service";
	protected static final String MONGO_COLLECTION_NAME = "TAB_MAPENGINE_DATASERVICE_INFO";


	public DataServiceManager() {
		super("id", new DataServiceBuilder(), MANAGER_ROOT);

		
	}

	public void init(){
		super.init();
		this.beanBuilder.setClient(client);
		if (client.getZoo() != null) {
			this.zkDao = new DataServiceZkDao(idFieldName, beanBuilder, table, client, managerRoot, readOnly);
			this.zkDao.init();
			try {
				this.zkDao.syncFromZK();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		this.mongoDao = new DataServiceMongoDao(client.initMongoClient(),  MONGO_COLLECTION_NAME, "id", beanBuilder);

	}

	public void register(IDataService dataService)  throws ZKException{
		
		this.checkInvalid(dataService);
		
		super.doAdd(dataService);
	}
	
	protected void checkInvalid(IDataService dataService) throws InvalidException{
		@SuppressWarnings("unused")
		ITable table = null;
		try {
			table = dataService.getTable();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			UException.printStackTrace(e);
			throw new InvalidException("dataService",dataService.mark() + "获得连接失败 ,错误原因:" + e.getMessage());
		}
		String[] aliasFields = dataService.getAliasFiledArr();
		Set<String> fieldsSet = new HashSet<>();
		Set<String> duplicateSet = new HashSet<>();
		int len = aliasFields.length;
		for (int i = 0; i < len; i++) {
			if (fieldsSet.contains(aliasFields[i])){
				duplicateSet.add(aliasFields[i]);
			}else {
				fieldsSet.add(aliasFields[i]);
			}
		}
		if (!duplicateSet.isEmpty()){
			String duplicateStr = StringUtils.join(duplicateSet,",");
			throw new InvalidException("dataService","字段: " + duplicateStr + " 存在别名冲突，请勿选择存在别名冲突的字段");
		}

	}
	
	//vector_service
	

	public void unRegister(String id) throws ZKException {
		super.doRemove(id);
	}

	
	public void update(IDataService dataService) throws ZKException  {
		
		this.checkInvalid(dataService);

		//更新时，清空字段信息，重新获取
		dataService.getBean().setFields(null);
		dataService.getFields();
		
		super.doUpdate(dataService);

		List<IVectorService> relatedServices = this.findRelatedVectorService(dataService);

		this.updateRelatedVectorService(relatedServices);
	}

	private List<IVectorService> findRelatedVectorService(IDataService dataService) {
		IVectorServiceManager vectorServiceManager = client.getVectorServiceManager();
		List<IVectorService> services = vectorServiceManager.list(new QueryFilter());
		List<IVectorService> relatedServices = services.stream().filter(service -> {
			Iterator<ILayer> it = service.getLayerMap().values().iterator();
			while (it.hasNext()){
				ILayer layer = it.next();
				Iterator<ILayerLevel> levelIterator = layer.getLevelMap().values().iterator();
				while (levelIterator.hasNext()){
					ILayerLevel level = levelIterator.next();
					if (dataService.getId().equals(level.getDataSet().getDataService().getId())){
						return true;
					}
				}
			}
			return false;
		}).collect(Collectors.toList());;
		return relatedServices;
	}

	private List<IStyleService> findRelatedStyle(List<IVectorService> services){
		List<IStyleService> styles = new ArrayList<>();
		IStyleManager styleManager = client.getStyleManager();
		for (int i = 0; i < services.size(); i++) {
			IVectorService service = services.get(i);
			QueryFilter filter = new QueryFilter();
			filter.addFilter("ownServer", Constants.OPERATION.EQ, service.getId());
			List<IStyleService> stylesList = styleManager.list(filter);
			styles.addAll(stylesList);
		}
		return styles;
	}

	private void updateRelatedVectorService(List<IVectorService> relatedServices) throws ZKException   {
		IVectorServiceManager vectorServiceManager = client.getVectorServiceManager();
		for(IVectorService service : relatedServices){
			IVectorService newService = new VectorServiceBuilder(service.getClient()).fromDataService(service.getId(), service.getDataServiceIds(), service.getOrigin(), service.getResolutions());
			if(service.getStorageInfo() != null){
				newService.addStorageInfo(service.getStorageInfo());
			}
			if(service.getUtfgridInfo() != null){
				newService.addUtfgridInfo(service.getUtfgridInfo());
			}
			if(service.getCacheInfo() != null){
				newService.addCacheInfo(service.getCacheInfo());
			}
			vectorServiceManager.update(newService);
		}
	}

	private void updateRelatedStyle(List<IStyleService> relatedStyles, String oldName, String newName) throws RollbackException, ZKException  {
		IStyleManager styleManager = client.getStyleManager();
		
		boolean allPass = true;
		ZKException zkException = null;
		for(IStyleService style : relatedStyles){
			StyleJson styleJson = style.getStyleJsonBean();
			this.updateLayers(styleJson.getLayers(), oldName, newName);
			style.setStyleJson(styleJson.toJson());
			try {
				styleManager.save(style);
			} catch (ZKException e) {
				// TODO Auto-generated catch block
				UException.printStackTrace(e);
				allPass = false;
				zkException = e;
				break;
			}
		}
		if(!allPass){
			//尝试回滚,当然也会失败
			for(IStyleService style : relatedStyles){
				StyleJson styleJson = style.getStyleJsonBean();
				this.updateLayers(styleJson.getLayers(), newName, oldName);
				style.setStyleJson(styleJson.toJson());
				try {
					styleManager.save(style);
				} catch (ZKException e) {
					// TODO Auto-generated catch block
					UException.printStackTrace(e);
					allPass = false;
					throw new RollbackException(e);
				}
			}
			
			
			throw new ZKException(zkException.getMessage());
		}
		
		
		
	}

	private void updateLayers(List<AbstractLayer> layers, String oldName, String newName){
		int size = layers.size();
		for (int i = 0; i < size; i++) {
			AbstractLayer layer = layers.get(i);
			if (layer instanceof Layer){
				if (((Layer) layer).getData().equals(oldName)){
					((Layer) layer).setData(newName);
				}else if (((Layer) layer).getData().equals(oldName + LayerBean.LABEL_SUFFIX)){
					((Layer) layer).setData(newName + LayerBean.LABEL_SUFFIX);
				}
			}else {
				this.updateLayers(layer.getChildren(), oldName, newName);
			}
		}
	}


	public IScroll<IDataService> scroll(QueryFilter queryFilter){
		return super.scroll(queryFilter);
	}


	public void start(String id) throws ZKException {
		super.start(id);
	}

	
	public void stop(String id) throws ZKException {
		super.stop(id);
	}

	@Override
	public IDataService get(String id) {
		// TODO Auto-generated method stub
		return super.get(id);
	}



	@Override
	public IDataService findOne(QueryFilter queryFilter) {
		// TODO Auto-generated method stub
		queryFilter.setStart(0);
		queryFilter.setLimit(1);
		List<IDataService> iList = super.list(queryFilter);
		if(iList == null || iList.isEmpty()){
			return null;
		}else{
			return iList.get(0);
		}
	}

	@Override
	public void rename(String id, String newName) throws ZKException {
		IDataService dataService = this.get(id);
		if (dataService == null){
			throw new RuntimeException("数据服务 " + id + " 不存在");
		}
		String oldName = dataService.getName();
		dataService.getBean().setName(newName);

		IStorageInfo storageInfo = dataService.getStorageInfo();
		if(storageInfo != null){
			storageInfo.rename(dataService.getStorageName());
			if (storageInfo.getLabelStorageName() != null){
				storageInfo.renameLabel(dataService.getLabelStorageName());
			}
		}

		List<IVectorService> relatedServices = this.findRelatedVectorService(dataService);
		List<IStyleService> relatedStyles = this.findRelatedStyle(relatedServices);

		//更新时，清空字段信息，重新获取
		dataService.getBean().setFields(null);
		dataService.getFields();

		this.doUpdate(dataService);

		this.updateRelatedVectorService(relatedServices);
		try {
			this.updateRelatedStyle(relatedStyles, oldName, newName);
		} catch (RollbackException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			String msg = "严重错误 dataService id : " + id + " 更改名称回滚失败 错误 : " + e.getMessage();
			throw new ZKException(msg);
		}

	}


}
