package com.northpool.service.manager.style;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.mongodb.BasicDBObject;
import com.northpool.service.config.style.IStyleService;
import com.northpool.service.config.style.StyleBuilder;
import com.northpool.service.dao.IMongoDao;
import com.northpool.service.dao.style.StyleMongoDao;
import com.northpool.service.dao.style.StyleZkDao;
import com.northpool.service.manager.abstractclass.AbstractManager;
import com.northpool.service.manager.abstractclass.NotFoundException;
import com.northpool.service.manager.abstractclass.ZKException;
import com.northpool.service.mapserver.MapServerAccess;
import com.northpool.service.mapserver.MapserverAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;
import java.util.concurrent.CompletionException;

public class StyleManager extends AbstractManager<IStyleService, StyleBuilder> implements IStyleManager {

    Logger logger = LoggerFactory.getLogger(StringBuilder.class);

    protected static final String MANAGER_ROOT = "style";

    protected static final String MONGO_COLLECTION_NAME = "TAB_MAPENGINE_STYLE_INFO";

    protected MapServerAccess mapServerAccess;


    public StyleManager(){
        super("uuid", StyleBuilder.getInstance(),  MANAGER_ROOT);

    }

    public void init(){
        super.init();
        this.mongoDao = new StyleMongoDao(client.initMongoClient(),  MONGO_COLLECTION_NAME, this.idFieldName, beanBuilder);
        this.mapServerAccess = client.getMapServerAccess();
        if (client.getZoo() != null){
            this.zkDao = new StyleZkDao(idFieldName, beanBuilder, table, client, managerRoot, readOnly, this);
            this.zkDao.init();
            try {
                this.zkDao.syncFromZK();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }


    @Override
    public void save(IStyleService style) throws ZKException  {
        IStyleService finalStyle = style;
        String result = null;
        try {
            result = MapServerAccess.saveStyle(finalStyle.getOwnServer(), finalStyle.getName(), finalStyle.getStyleJson(), finalStyle.getXmin(), finalStyle.getYmin(), finalStyle.getXmax(), finalStyle.getYmax());
        } catch (MapserverAccessException e) {
            throw new CompletionException(e);
        }
        logger.debug(result);
        JSONObject resultJson = JSON.parseObject(result);
        if (!resultJson.containsKey("success") || !resultJson.get("success").equals(true)
                || !resultJson.containsKey("id")) {
            throw new CompletionException(new StyleInValidException(resultJson.getString("message")));
        }

        BasicDBObject filter = new BasicDBObject();
        filter.put("name", style.getName());
        filter.put("ownServer", style.getOwnServer());
        style = this.mongoDao.findone(filter);
        int count = 0;
        int maxTry = 3;
        while (style == null && count < maxTry){//防止mongo查询时，引擎还没完成写入，引擎明明是同步接口，为啥会有这种sb问题....后续再找
            style = this.mongoDao.findone(filter);
            try {
				Thread.sleep(50);
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				//e.printStackTrace();
				//不需要处理
				if(count == maxTry){
					e.printStackTrace();
					throw new ZKException("获得mongodb数据超过最大尝试次数");
				}
			}
            count++;
        }

        if(this.get(style.getId()) == null){//引擎转换为js后，保存到mongo，client从mongo中取出，然后同步到zk
            if (this.zkDao != null){
                this.saveBeanToHashTable(style);
                try {
                    this.zkDao.saveBeanTozk(style);
                } catch (Exception e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                    this.removeBeanToHashTable(style.getId());
                    throw new ZKException(e);
                }
            }
        }else {
            if (this.zkDao != null){
                this.table.update(style);
                this.zkDao.updateBeanTozk(style);
            }
        }
    }

    @Override
    public void remove(String[] ids) throws  ZKException {
        for (int i = 0; i < ids.length; i++) {
            try{
                super.doRemove(ids[i]);
            }catch (NotFoundException e){//多个样式，某个样式找不到，不影响其他的删除
                logger.info(e.getMessage());
                continue;
            }
        }
    }

    @Override
    public IStyleService get(String belongService, String styleId) {
        return super.get(belongService + "_" + styleId);
    }

	@Override
    public boolean checkVersion(String belongService, String styleId, String version) {
        IStyleService style = this.get(belongService, styleId);
        if(style != null) {
            return Objects.equals(version, style.getVersion());
        }
        return false;
    }

    @Override
    public void update(IStyleService style) throws ZKException  {
        this.remove(new String[]{style.getUuid()});
        style.getBean().setUuid(style.getOwnServer() + "_" + style.getName());
        this.doAdd(style);
    }

    public IMongoDao<IStyleService> getMongoDao(){
        return this.mongoDao;
    };
    
}
