/*
 * Copyright 2012 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.northpool.service.netty.websocket;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.northpool.commons.reflect.Bean;
import com.northpool.service.client.Client;
import com.northpool.service.manager.IClientManager;
import com.northpool.service.manager.cell.ITileDataCellManager;
import com.northpool.service.manager.data_service.IDataServiceManager;
import com.northpool.service.manager.data_sources.IDataSourcesManager;
import com.northpool.service.manager.font.IFontManager;
import com.northpool.service.manager.style.IStyleManager;
import com.northpool.service.manager.texture.ITextureManager;
import com.northpool.service.manager.vector_service.IVectorServiceManager;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.websocketx.BinaryWebSocketFrame;
import io.netty.handler.codec.http.websocketx.TextWebSocketFrame;
import io.netty.handler.codec.http.websocketx.WebSocketFrame;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;

/**
 * Echoes uppercase content of text frames.
 */
public class WebSocketFrameHandler extends SimpleChannelInboundHandler<WebSocketFrame> {


    private static final Logger logger = LoggerFactory.getLogger(WebSocketFrameHandler.class);
    private Client client;

    private HashMap<String, Method> methodMap;
    private HashMap<String, IClientManager> managerMap;

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, WebSocketFrame frame) throws Exception {
        // ping and pong frames already handlede
        byte[] markByte = null;
        
        
     
        
        try{
            if (frame instanceof TextWebSocketFrame) {
                String str = ((TextWebSocketFrame) frame).text();
                JSONArray arr = JSON.parseArray(str);
                List<Object> params = arr.subList(0, arr.size() - 2);

                String markId = (String) arr.get(arr.size() - 1);
                String methodKey = (String) arr.get(arr.size() - 2);
                String managerName = methodKey.substring(0, methodKey.indexOf("_"));
                IClientManager manager = this.managerMap.get(managerName);
                

                Method method = this.methodMap.get(methodKey);
         

                markByte = markId.getBytes("utf-8");
                if (method.getReturnType().getName().equals("void")){
                    params.add(ctx);
                    params.add(markByte);
                }

                Object result = null;
      
                if (params.size() == 0){
                    result = method.invoke(manager);
                }else {
                     try{
                         result = method.invoke(manager, params.stream().toArray(Object[]::new));
                     }catch(Exception e){
                         e.printStackTrace();
                         System.out.println("error" + e.getMessage());
                     }
                }
                if (method.getReturnType().getName().equals("void")){
                    return;
                }
                byte[] bytedata = new byte[0];
                if (result instanceof byte[]){
                    bytedata = (byte[]) result;
                }else if (result instanceof String){
                    bytedata = ((String)result).getBytes("utf-8");
                }
               
                ctx.writeAndFlush(new BinaryWebSocketFrame(Unpooled.wrappedBuffer(markByte, new byte[]{1}, bytedata)));
                
            } else {
                String message = "unsupported frame type: " + frame.getClass().getName();
                throw new UnsupportedOperationException(message);
            }
        }catch (Exception e){
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            String stackTrace = null;
            try {
                e.printStackTrace(pw);
                stackTrace = sw.toString();
            } finally {
                pw.close();
            }
            ctx.writeAndFlush(new BinaryWebSocketFrame(Unpooled.wrappedBuffer(markByte, new byte[]{0}, stackTrace.getBytes("utf-8"))));
        }
     

    }

    public void setClient(Client client){
        this.client = client;
    }

    public void init(){
    	 try{
        methodMap = new HashMap<>();
        managerMap = new HashMap<>();
        IDataSourcesManager dataSourcesManager = client.getDataSourcesManager();
        managerMap.put("IDataSourcesManager", dataSourcesManager);
        Method[] methods = Bean.getDeclaredMethods(IDataSourcesManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("IDataSourcesManager_" + method.getName(), method);
        }

        IDataServiceManager dataServiceManager = client.getDataServiceManager();
        managerMap.put("IDataServiceManager", dataServiceManager);
        methods = Bean.getDeclaredMethods(IDataServiceManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("IDataServiceManager_"+method.getName(), method);
        }

        IVectorServiceManager vectorServiceManager = client.getVectorServiceManager();
        managerMap.put("IVectorServiceManager", vectorServiceManager);
        methods = Bean.getDeclaredMethods(IVectorServiceManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
          
            methodMap.put("IVectorServiceManager_"+method.getName(), method);
        }


        IStyleManager styleManager = client.getStyleManager();
        managerMap.put("IStyleManager", styleManager);


        methods = Bean.getDeclaredMethods(IStyleManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("IStyleManager_"+method.getName(), method);
        }

        ITextureManager textureManager = client.getTextureManager();
        managerMap.put("ITextureManager", textureManager);

        methods = Bean.getDeclaredMethods(ITextureManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("ITextureManager_"+method.getName(), method);
        }


        IFontManager fontManager = client.getFontManager();
       
        managerMap.put("IFontManager", fontManager);

        methods = Bean.getDeclaredMethods(IFontManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("IFontManager_"+method.getName(), method);
        }

        ITileDataCellManager tileDataCellManager = client.getTileDataCellManager();
        managerMap.put("ITileDataCellManager", tileDataCellManager);

        methods = Bean.getDeclaredMethods(ITileDataCellManager.class, null);

        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            methodMap.put("ITileDataCellManager_"+method.getName(), method);
        }
    	 }catch(Exception e){
    		 e.printStackTrace();
    	 }
       
    }
    
   
	
	
	

    
    
   /* public static void main(String[] aaa){
    	Method[] methods = Bean.getDeclaredMethods(IDataServiceManager.class, null);
    	System.out.println(methods);
    	
    }*/

}
