package com.northpool.service.config.vector_service;


import com.alibaba.fastjson.JSON;
import com.northpool.bean.JsonableBuilder;
import com.northpool.resources.datatable.SpatialInfo;
import com.northpool.service.client.Client;
import com.northpool.service.config.Constants;
import com.northpool.service.config.data_service.IDataService;
import com.northpool.service.config.vector_service.dataset.DataSetBean;
import com.northpool.service.config.vector_service.exception.*;
import com.northpool.service.config.vector_service.layer.Label;
import com.northpool.service.config.vector_service.layer.LayerBean;
import com.northpool.service.config.vector_service.layer.LayerLevelBean;
import com.northpool.service.exception.LoadXmlException;
import com.northpool.service.manager.abstractclass.DocumentableBuilder;
import com.northpool.service.xmlloader.vectorservice.VectorServiceLoader;
import com.northpool.spatial.grid.Constants.GRID_TYPE;
import com.northpool.spatial.grid.Constants.GRID_UNIT;
import com.northpool.spatial.grid.Grid;
import com.northpool.spatial.grid.GridManager;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

import static com.northpool.spatial.Constants.GEO_TYPE.*;

public class VectorServiceBuilder  implements JsonableBuilder<IVectorService>, DocumentableBuilder<IVectorService> {

	private Client client;
	public VectorServiceBuilder(){

	}

	public VectorServiceBuilder(Client client){
		this.client = client;
	}

	/*VectorServiceBuilder(Client client){
		this.client = client;
	}
	
	public static VectorServiceBuilder getInstance(Client client){
		return new VectorServiceBuilder(client);
	}*/

	@Override
	public IVectorService fromJson(String str) throws Exception {
		// TODO Auto-generated method stub
		VectorServiceBean vectorService = null;
		try{
			vectorService = JSON.parseObject(str, VectorServiceBean.class);
		}catch(Exception e){
			throw new Exception(String.format("解析JSON失败:%s 不能翻译成vectorService", str));
		}
		if (vectorService.getId() == null){//没有id的认为是异常数据，返回空
			return null;
		}
		IVectorService vectorServiceShell = new VectorServiceShell(this.client,vectorService);
		return vectorServiceShell;
	}

	@Override
	public IVectorService fromDocument(Document str) throws Exception {
		str.remove("_id");
		return this.fromJson(str.toJson());
	}

	public IVectorService fromXmlInputStream(InputStream in) throws LoadXmlException {
		VectorServiceLoader xml = new VectorServiceLoader(client);
		VectorServiceBean service = xml.load(in);
		service.setVersion(new Date().getTime() + "");
		IVectorService VectorServiceShell = new VectorServiceShell(this.client,service);
		return VectorServiceShell;
	}


	public IVectorService fromDataService(String serverName, List<String> dataServiceIds){
		return this.fromDataService(serverName, dataServiceIds, null, null);
	}

	public IVectorService fromDataService(String serverName, List<String> dataServiceIds, int[] origin, double[] resolutions){
		if (dataServiceIds == null || dataServiceIds.isEmpty()){
			return null;
		}
		if (StringUtils.isEmpty(serverName)){
			throw new VectorServiceNameInvalidException(serverName);
		}
		VectorServiceBean vectorService = new VectorServiceBean();
		vectorService.setId(serverName);
		vectorService.setVersion(new Date().getTime()+"");

		//查询dataservice
		List<IDataService> dataServices = dataServiceIds.stream().map(dataServiceId -> {
			IDataService dataService = client.getDataServiceManager().get(dataServiceId);
			if (dataService == null){
				throw new DataServiceNotFoundException(dataServiceId);
			}
			return dataService;
		}).collect(Collectors.toList());

		//检查空间参考，检查重名
		Set<Integer> sridSet = new HashSet<>();
		Set<String> namesSet = new HashSet<>();
		Map<String, SpatialInfo> spatialInfoMap = new HashMap<>();
		for (int i = 0; i < dataServices.size(); i++) {
			IDataService dataService = dataServices.get(i);
			if (namesSet.contains(dataService.getName())){
				throw new DataServiceNameDuplicateException(dataService.getName());
			}
			namesSet.add(dataService.getName());
			SpatialInfo info = null;
			try {
				info = dataService.getTable().spatialInfo();
			} catch (Exception e) {
				throw new LayerCreateException(e);
			}
			spatialInfoMap.put(dataService.getName(), info);
			sridSet.add(info.getSrid());
		}
		if (sridSet.size() > 1 && sridSet.contains(3857)){
			throw new SpatialReferenceConflictException(sridSet);
		}

		GRID_TYPE gridType = GRID_TYPE.tdt;
		if (origin != null && resolutions != null){
			gridType = GRID_TYPE.esri;
		}

		//获取格网信息
		GRID_UNIT gridBase = GRID_UNIT.degree;
		if (sridSet.contains(3857)){
			gridBase = GRID_UNIT.meter;
		}
		String gridTreeName = gridType.name() + "_" + gridBase.name() + "_base512";

		vectorService.setGridTreeName(gridTreeName);
		vectorService.setResolutions(resolutions);
		vectorService.setOrigin(origin);

		Grid grid = GridManager.getQuadtreeGrid(gridTreeName);

		Integer startLevel = grid.getBeginLevel();

		Integer endLevel = LayerBean.MAX_LEVEL;

		// 解析数据集
		Map<String, DataSetBean> dataSetMap = dataServices.stream().map(dataService -> {
			DataSetBean dataSet = new DataSetBean();
			dataSet.setId(dataService.getName());
			dataSet.setDataServiceId(dataService.getId());
			dataSet.setFilter(dataService.getBean().getFilter());
			return dataSet;
		}).collect(Collectors.toMap(dataSetBean -> dataSetBean.getId(), dataSetBean -> dataSetBean));

		vectorService.setDataSetMap(dataSetMap);

		// 解析图层
		Map<String, LayerBean> layerMap = this.getLayerMap(dataServices, dataSetMap, startLevel, endLevel);

		vectorService.setLayerMap(layerMap);

		IVectorService VectorServiceShell = new VectorServiceShell(this.client,vectorService);

		return VectorServiceShell;
	}

	private Map<String, LayerBean> getLayerMap(List<IDataService> dataServices, Map<String, DataSetBean> dataSetMap, Integer beginLevel, Integer endLevel) {
		Map<String, LayerBean> layerMap = dataServices.stream().map(dataService -> {
			try {
				return this.createLayer(dataService, dataSetMap.get(dataService.getName()), beginLevel, endLevel);
			} catch (Exception e) {
				throw new LayerCreateException(dataService.getName());
			}
		}).collect(Collectors.toMap(layerBean -> layerBean.getId(), layerBean -> layerBean));
		return layerMap;
	}

	private LayerBean createLayer(IDataService dataService, DataSetBean dataSet, Integer beginLevel, Integer endLevel)  {
		LayerBean layer = new LayerBean();
		layer.setBeginLevel(beginLevel);
		layer.setEndLevel(endLevel);
		layer.setId(dataService.getName());
		com.northpool.spatial.Constants.GEO_TYPE type = dataService.getGeoType();
		Map<Integer, LayerLevelBean> allLayerLevel = new HashMap<Integer,LayerLevelBean>();
		for (Integer i = layer.getBeginLevel(); i <= layer.getEndLevel(); i++) {
			LayerLevelBean layerLevel = new LayerLevelBean();
			layerLevel.setDataSetId(dataSet.getId());
			layerLevel.setLevel(i);
			allLayerLevel.put(i, layerLevel);
		}

		//补充LayerLevel使用的dataset,如果有其他公用字段，也需要以同补充
		// 加入标注
		layer.setLevelMap(allLayerLevel);
		if (((type == LINESTRING || type == MULTILINESTRING) && dataService.getBean().getAdvancedOptions().isHasLineLabel())
				||((type == POLYGON || type == MULTIPOLYGON) && dataService.getBean().getAdvancedOptions().isHasPolygonLabel())
				|| type == MULTIPOINT || type == POINT){
			layer.setHaveLabel(true);
			Label label = new Label();
			if(type == LINESTRING || type == MULTILINESTRING){
				label.setType(Constants.LABEL_TYPE.line);
			}else if(type == MULTIPOINT || type == POINT){
				label.setType(Constants.LABEL_TYPE.point);
			}else if(type == POLYGON || type == MULTIPOLYGON){
				label.setType(Constants.LABEL_TYPE.area);
			}
			layer.setLabel(label);
		}
		return layer;
	}

	public void setClient(Client client) {
		this.client = client;
	}
}
