package com.northpool.service.manager.font;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.northpool.exception.NotSupport;
import com.northpool.resources.command.Constants.OPERATION;
import com.northpool.resources.command.QueryFilter;
import com.northpool.service.config.Constants;
import com.northpool.service.config.font.FontBean;
import com.northpool.service.config.font.FontBuilder;
import com.northpool.service.config.font.FontShell;
import com.northpool.service.config.font.IFontService;
import com.northpool.service.dao.IMetaDataDao;
import com.northpool.service.dao.font.FontIgniteDao;
import com.northpool.service.dao.font.FontMongoDao;
import com.northpool.service.dao.font.FontZkDao;
import com.northpool.service.manager.abstractclass.AbstractManager;
import com.northpool.service.manager.abstractclass.ZKException;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;

import java.util.*;
import java.util.stream.Collectors;

public class FontManager extends AbstractManager<IFontService, FontBuilder> implements IFontManager {

    protected static final String MANAGER_ROOT = "font";


    FontMongoDao mongoGridFSDao;

    public FontManager(){
        super(FontBuilder.getInstance(), MANAGER_ROOT);
        this.idFieldName = "filename";
    }

    public void init() throws Exception {

        super.init();
        if (this.metaDataDao instanceof FontZkDao){
            this.mongoGridFSDao = (FontMongoDao) ((FontZkDao) this.metaDataDao).getMongoDao();
        }


    }

    @Override
    protected IMetaDataDao<IFontService> getMetaDataDao() throws Exception {
        if(this.metaDataDao == null){
            if (client.getZoo() != null) {
                FontZkDao metaDao = new FontZkDao(idFieldName, beanBuilder, table, client, managerRoot, readOnly, this);
                metaDao.init();
                this.metaDataDao = metaDao;
            }else if (client.getIgnite() != null){
                FontIgniteDao igniteDao = new FontIgniteDao(idFieldName, table, client, managerRoot, readOnly, client.getIgnite(), this);
                igniteDao.init();
                this.metaDataDao = igniteDao;
            }
        }
        return this.metaDataDao;
    }


    @Override
    public void register(IFontService font) throws Exception {
        if(this.get(font.getId()) == null){
            super.doAdd(font);
        }else {
            super.doUpdate(font);
        }
    }

    @Override
    
    public void unRegister(String id) throws Exception {
        super.doRemove(id);
    }

    @Override
    public void unRegister(String family, Constants.FONT_STYLE style, Constants.FONT_WEIGHT weight, Constants.FONT_FILE_TYPE fileType) throws Exception {
        String id = this.generateFontId(family, style, weight, fileType);
        super.doRemove(id);

    }

    @Override
    public void unRegisterByFamily(String family) throws Exception {
        if (StringUtils.isEmpty(family)){
            return;
        }
        QueryFilter filter = new QueryFilter();
        filter.addFilter("family", OPERATION.EQ, family);
        List<IFontService> list = this.list(filter);
        for (IFontService iFontService : list) {
            this.unRegister(iFontService.getId());
        }
    }
    public List<IFontService> list(QueryFilter filter) {
        List<IFontService> list = super.list(filter);
        
        if(list == null) {
            list = new ArrayList<IFontService>(0);
        }
        
        Map<String, List<IFontService>>  map = list.stream().collect(Collectors.groupingBy(font -> font.getFamily() + "_" + font.getStyle() + "_" + font.getWeight()));
        Iterator<String> iterator = map.keySet().iterator();
        while (iterator.hasNext()){
            String key = iterator.next();
            List<IFontService> flist = map.get(key);
            boolean hasNN = flist.stream().filter(font -> font.getStyle() == Constants.FONT_STYLE.normal && font.getWeight() == Constants.FONT_WEIGHT.normal).findFirst().isPresent();
            if (!hasNN){
                list.removeAll(flist);
            }

        }
        
        return list;
    }

    /**
     * 获取字体树
     * @param idPatternStr 用于根据id模糊查询，可以为空
     * @param start
     * @param limit
     * @return
     */
    
    @Override
    public String map(String idPatternStr, Integer start, Integer limit){
        QueryFilter filter = new QueryFilter();
        if (StringUtils.isNotEmpty(idPatternStr)){
            String[] idPatterns = idPatternStr.split(",");
            for (int i = 0; i < idPatterns.length; i++) {
                String idPattern = idPatterns[i];
                if (i == 0){
                    filter.addFilter("id", com.northpool.resources.command.Constants.OPERATION.LFK, idPattern);
                }else {
                    filter.addFilter(com.northpool.resources.command.Constants.RELATION.OR, "id", com.northpool.resources.command.Constants.OPERATION.LFK, idPattern);
                }
            }
        }

        List<IFontService> list = this.list(filter);
        Map<String, List<IFontService>> map =  list.stream().collect(Collectors.groupingBy(font -> font.getFamily()));
        Iterator<String> iterator = map.keySet().iterator();
        JSONObject obj = new JSONObject();
        while (iterator.hasNext()){
            String family = iterator.next();
            List<IFontService> familyList = map.get(family);
            Map<String, Map<String, Object>> familymap = familyList.stream().collect(Collectors.groupingBy(font -> font.getStyle() +"_" +font.getWeight(),Collectors.toMap(font->font.getFileType().toString(), font -> font)));
            obj.put(family, familymap);
        }
        return obj.toJSONString();
    }

    
    @Override
    public IFontService get(String family, Constants.FONT_STYLE style, Constants.FONT_WEIGHT weight, Constants.FONT_FILE_TYPE fileType) {
        String id = this.generateFontId(family, style, weight, fileType);
        IFontService font = super.get(id);
        return font;
    }

    @Override
    public boolean checkVersion(String family, Constants.FONT_STYLE style, Constants.FONT_WEIGHT weight, String version, Constants.FONT_FILE_TYPE fileType) {
        IFontService font = this.get(family, style, weight, fileType);
        return Objects.equals(version, font.getVersion());
    }

    @Override
    public boolean checkVersion(String id, String version) {
        IFontService font = this.get(id);
        return Objects.equals(version, font.getVersion());
    }

    @Override
    public byte[] getContent(String id) {
        byte[] content = mongoGridFSDao.getContent(new Document(this.idFieldName, id));
        return content;
    }

    @Override
    public String getDefaultFont() throws Exception {
        List<IFontService> list = this.list(new QueryFilter());
        Optional<IFontService> optional = list.stream().filter(font -> font.isDefaultFont()).findFirst();
        if (optional.isPresent()){
            return optional.get().getFamily();
        }else {
            return null;
        }
    }

    @Override
    public String getForeendUnCommon() throws Exception {
        List<IFontService> list = this.list(new QueryFilter());
        Optional<IFontService> optional = list.stream().filter(font -> font.isForeendUnCommon()).findFirst();
        if (optional.isPresent()){
            return optional.get().getFamily();
        }else {
            return null;
        }
    }

    @Override
    public String getBackendUnCommon() throws Exception {
        List<IFontService> list = this.list(new QueryFilter());
        Optional<IFontService> optional = list.stream().filter(font -> font.isBackendUnCommon()).findFirst();
        if (optional.isPresent()){
            return optional.get().getFamily();
        }else {
            return null;
        }
    }

    @Override
    public String getFontConfig() throws Exception {
        List<IFontService> list = this.list(new QueryFilter());
        Optional<IFontService> optional = list.stream().filter(font -> font.isForeendUnCommon()).findFirst();
        String foreendUnCommon = "";
        if (optional.isPresent()){
            foreendUnCommon = optional.get().getFamily();
        }
        Set<String> families = list.stream().map(font -> font.getFamily()).collect(Collectors.toSet());
        Map<String, Object> result = new HashMap<>(2);
        result.put("foreendUnCommonFont", foreendUnCommon);
        result.put("fonts", families);
        return JSON.toJSONString(result);
    }


    private String generateFontId(String family, Constants.FONT_STYLE style, Constants.FONT_WEIGHT weight, Constants.FONT_FILE_TYPE fileType){
        return family + "_" + style + "_" + weight + "." + fileType;
    }

	@Override
	public void update(IFontService t) throws ZKException {
		// TODO Auto-generated method stub
		throw new NotSupport();
	}

	@Override
	public void start(String id) throws ZKException {
		// TODO Auto-generated method stub
		throw new NotSupport();
	}

	@Override
	public void stop(String id) throws ZKException {
		// TODO Auto-generated method stub
		throw new NotSupport();
	}

	@Override
	public IFontService findOne(QueryFilter queryFilter) {
		// TODO Auto-generated method stub
		throw new NotSupport();
	}

	@Override
	public void rename(String id, String newName) throws ZKException {
		// TODO Auto-generated method stub
		throw new NotSupport();
	}

    @Override
    protected void saveBeanToHashTable(IFontService fontService){
        FontBean clone = null;
        try {
            clone = fontService.getBean().clone();
        } catch (CloneNotSupportedException e) {
            e.printStackTrace();
        }
        clone.setContent(null);//内存中不保存字体文件
        super.saveBeanToHashTable(new FontShell(clone));

    }
}
