/**
　 * <p>Title: RasterServiceBuilder.java</p>
　 * <p>Description: </p>
　 * <p>Copyright: Copyright (c) 2021</p>
　 * <p>Company: northpool</p>
　 * @author chy
　 * @date 2021年12月22日
　 * @version 1.0
*/
package com.northpool.service.config.raster_service;

import com.alibaba.fastjson.JSON;
import com.northpool.bean.JsonableBuilder;
import com.northpool.service.client.Client;
import com.northpool.service.config.Constants.SERVICE_TYPE;
import com.northpool.service.config.data_service.raster.IRasterDataService;
import com.northpool.service.config.raster_service.dataset.RasterDataSetBean;
import com.northpool.service.config.raster_service.layer.RasterLayerBean;
import com.northpool.service.config.raster_service.layer.RasterLayerLevelBean;
import com.northpool.service.config.vector_service.exception.*;
import com.northpool.service.manager.abstractclass.DocumentableBuilder;
import com.northpool.spatial.Constants.LAYER_TYPE;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author chy
 *
 */
public class RasterServiceBuilder implements JsonableBuilder<IRasterService>, DocumentableBuilder<IRasterService> {

    protected Client client;
    
    public RasterServiceBuilder(){

    }

    public RasterServiceBuilder(Client client) {
        this.client = client;
    }
    
    public void setClient(Client client) {
        this.client = client;
    }
    
    @Override
    public IRasterService fromDocument(Document document) throws Exception {
        document.remove("_id");
        return this.fromJson(document.toJson());
    }

    @Override
    public IRasterService fromJson(String str) throws Exception {
        RasterServiceBean imageService = null;
        try {
            imageService = JSON.parseObject(str, RasterServiceBean.class);
            SERVICE_TYPE serviceType = imageService.getServiceType();
            if(serviceType != SERVICE_TYPE.image_service && serviceType != SERVICE_TYPE.terrain_service) {
                throw new Exception();
            }
        } catch(Exception e) {
            throw new Exception(String.format("解析JSON失败:%s 不能翻译成影像服务或地形服务", str));
        }
        IRasterService imageServiceShell = new RasterServiceShell(this.client, imageService);
        return imageServiceShell;
    }
    
    protected Integer getSrid(List<IRasterDataService> rasterDataServices) {
        Set<Integer> sridSet = new HashSet<>();
        Set<String> namesSet = new HashSet<>();
        
        for(IRasterDataService rasterDataService : rasterDataServices) {
            String dataServiceName = rasterDataService.getName();
            if(StringUtils.isBlank(dataServiceName)) {
                throw new NoDataServiceNameException();
            }
            if (namesSet.contains(dataServiceName)) {
                throw new DataServiceNameDuplicateException(dataServiceName);
            }
            namesSet.add(dataServiceName);
            Integer srid = rasterDataService.getSrid();

            if(srid == null) {
                throw new NoSpatialReferenceException();
            }
            
            if(!sridSet.contains(srid)) {
                sridSet.add(srid);
            }
            
            if(sridSet.size() > 1) {
                throw new SpatialReferenceConflictException(sridSet);
            }
        }
        
        return sridSet.iterator().next();
    }
    
    protected LinkedHashMap<String, RasterDataSetBean> getDataSetMap(List<IRasterDataService> rasterDataServices) {

        LinkedHashMap<String, RasterDataSetBean> rasterDataSetMap = rasterDataServices.stream().map(dataService -> {
            RasterDataSetBean dataSet = new RasterDataSetBean();
            dataSet.setId(dataService.getName());
            dataSet.setDataServiceId(dataService.getId());
            return dataSet;
        }).collect(Collectors.toMap(dataSetBean -> dataSetBean.getId(), dataSetBean -> dataSetBean, (x, y) -> y, LinkedHashMap::new));
        
        return rasterDataSetMap;
    }
    
    protected LinkedHashMap<String, RasterLayerBean> getLayerMap(List<IRasterDataService> rasterDataServices,
            LinkedHashMap<String, RasterDataSetBean> dataSetMap, Integer beginLevel, Integer endLevel) {
        
        LinkedHashMap<String, RasterLayerBean> rasterLayerMap = rasterDataServices.stream().map(dataService -> {
            try {
                return this.createRasterLayer(dataService, dataSetMap.get(dataService.getName()), beginLevel, endLevel);
            } catch (Exception e) {
                throw new LayerCreateException(dataService.getName());
            }
        }).collect(Collectors.toMap(layerBean -> layerBean.getId(), layerBean -> layerBean, (x, y) -> y, LinkedHashMap::new));
        
        return rasterLayerMap;
    }

    protected RasterLayerBean createRasterLayer(IRasterDataService dataService, RasterDataSetBean dataSet,
            Integer beginLevel, Integer endLevel) {
        RasterLayerBean layer = new RasterLayerBean();
        layer.setBeginLevel(beginLevel);
        layer.setEndLevel(endLevel);
        layer.setId(dataService.getName());
        layer.setLayerType(LAYER_TYPE.RASTER);
        Map<Integer, RasterLayerLevelBean> allLayerLevel = new HashMap<Integer, RasterLayerLevelBean>();
        for (Integer i = layer.getBeginLevel(); i <= layer.getEndLevel(); i++) {
            RasterLayerLevelBean layerLevel = new RasterLayerLevelBean();
            layerLevel.setDataSetId(dataSet.getId());
            layerLevel.setLevel(i);
            allLayerLevel.put(i, layerLevel);
        }

        //补充LayerLevel使用的dataset,如果有其他公用字段，也需要以同补充
        layer.setLevelMap(allLayerLevel);
        
        return layer;
    }
}
