/**
　 * <p>Title: TerrainServiceLoader.java</p>
　 * <p>Description: </p>
　 * <p>Copyright: Copyright (c) 2021</p>
　 * <p>Company: northpool</p>
　 * @author chy
　 * @date 2021年12月23日
　 * @version 1.0
*/
package com.northpool.service.xmlloader.raster_service;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import com.northpool.resources.command.QueryFilter;
import com.northpool.resources.command.Constants.OPERATION;
import com.northpool.resources.datasource.IDataSource;
import com.northpool.service.client.Client;
import com.northpool.service.config.data_source.DataScourceShell;
import com.northpool.service.config.data_source.IDataSourceInService;
import com.northpool.service.config.raster_service.RasterServiceBean;
import com.northpool.service.config.raster_service.dataset.RasterDataSetBean;
import com.northpool.service.config.raster_service.layer.RasterLayerBean;
import com.northpool.service.config.raster_service.layer.RasterLayerLevelBean;
import com.northpool.service.exception.LoadXmlException;
import com.northpool.service.exception.XmlDefException;
import com.northpool.service.manager.abstractclass.ZKException;
import com.northpool.service.manager.data_sources.IDataSourcesManager;
import com.northpool.service.xmlloader.XmlLoader;
import com.northpool.service.xmlloader.vectorservice.AbstractXmlLoader;
import com.northpool.spatial.grid.GridManager;

/**
 * @author chy
 *
 */
public class TerrainServiceLoader extends AbstractXmlLoader implements XmlLoader<RasterServiceBean> {
    
    public final static String VERSION = "v1.0";
    public final static String SERVICE_TYPE = "terrain_service";
    public final static String LAYER_SUFFIX = "_layer";
    public final static String STORAGE_PREFIX = "storage_";
    
    public TerrainServiceLoader(Client client) {
        super(client);
    }

    @Override
    public RasterServiceBean load(InputStream inXml) throws LoadXmlException {
        try {
            SAXReader reader = new SAXReader();
            Document document = reader.read(inXml);
            Element e = document.getRootElement();
            RasterServiceBean terrainService = null;
            if("vindex".equalsIgnoreCase(e.getName())) {
                Element item = (Element) e.element("TerrainServices").elements().get(0);
                terrainService = this.parseInfo(item);
            } else {
                 terrainService = this.parseInfo(e);
            }
            return terrainService;
        } catch (DocumentException e) {
            throw new LoadXmlException("加载XML出错" + e.getMessage());
        } catch(Exception e) {
            e.printStackTrace();
            throw new XmlDefException(e.getMessage());
        }
    }

    @Override
    public List<RasterServiceBean> loadList(InputStream inXMl) throws LoadXmlException {
        try {
            SAXReader reader = new SAXReader();
            Document document = reader.read(inXMl);
            Element root = document.getRootElement();
            @SuppressWarnings("unchecked")
            List<Element> items = root.element("TerrainServices").elements();
            List<RasterServiceBean> terrainServerRegisterBeanArr = new ArrayList<RasterServiceBean>();
            for (Element item : items) {
                RasterServiceBean terrainServerRegisterBean = this.parseInfo(item);
                terrainServerRegisterBeanArr.add(terrainServerRegisterBean);
            }
            return terrainServerRegisterBeanArr;
        } catch(Exception e) {
            e.printStackTrace();
            throw new LoadXmlException(e.getMessage());
        }
    }

    @Override
    protected String getValidateXMLPath() {
        return this.getClass().getResource("").getPath() + "terrain_service_register.xsd";
    }
    
    /**
     * 解析数据源
     * @param item
     * @return
     * @throws Exception
     */
    private RasterServiceBean parseInfo(Element item) throws Exception {
        RasterServiceBean terrainService = RasterServiceBean.initTerrain();
        this.getInfoByStr(item, terrainService, "Id");
        try{
            String gridTreeName = "";
            if (item.elementText("Grid") != null){
                gridTreeName = item.elementText("Grid");
            }else{
                String gridBase = "degree";
                if (item.elementText("GridUnit") != null){
                    gridBase = item.elementText("GridUnit");
                }
                gridTreeName = "tdt_" + gridBase + "_base512";
            }
            
            //checkGridTreeName
            
            if(GridManager.getQuadtreeGrid(gridTreeName) == null){
                throw new RuntimeException("没有找到网格 " + gridTreeName);
            }
            
            
            terrainService.setGridTreeName(gridTreeName);
            
            // 解析数据源
            Map<String, IDataSourceInService> dataSourceShellMap = this.getDataSourceMap(item);
            
            // 解析数据集
            Map<String, RasterDataSetBean> dataSetMap = this.getRasterDataSetMap(item, dataSourceShellMap);
            
            terrainService.setDataSetMap(dataSetMap);
            
            // 解析图层
            LinkedHashMap<String, RasterLayerBean> layerMap = this.getLayerMap(item, dataSetMap);
            
            terrainService.setLayerMap(layerMap);
    
            return terrainService;
        } catch(Exception e) {
            e.printStackTrace();
            throw new LoadXmlException(terrainService.getId() + "解析" + e.getMessage());
        }

    }
    
    /**
     * 解析数据源MAP
     * 
     * @param root
     * @return
     */
    protected Map<String, IDataSourceInService> getDataSourceMap(Element root) throws Exception {
        Map<String, IDataSourceInService> dataSourceMap = new HashMap<String, IDataSourceInService>();
        @SuppressWarnings("unchecked")
        List<Element> items = root.element("DataSources").elements();
        for (Element item : items) {
            IDataSource dataSource = this.createDataSource(item);
            String idInXML = dataSource.getId();
            IDataSourceInService dataScourceShell = this.autoRegisterDataSource(dataSource);
            dataSourceMap.put(idInXML, dataScourceShell);
        }
        return dataSourceMap;
    }
    
    /**
     * 自动注册数据源（如果存在就不再注册）
     * @param dataSource
     * @return
     */
    protected IDataSourceInService autoRegisterDataSource(IDataSource dataSource) throws Exception {
        String mark = dataSource.mark();
        QueryFilter queryFilter = new QueryFilter();
        queryFilter.addFilter("mark", OPERATION.EQ, mark);
        IDataSourcesManager dataSourcesManager = this.client.getDataSourcesManager();
        IDataSourceInService dataScourceShell = dataSourcesManager.findOne(queryFilter);
        if(dataScourceShell == null) {
            String id = "autocreate_" + mark;
            
            dataScourceShell = new DataScourceShell(this.client, id, id, dataSource);
            if (dataSourcesManager.get(id) != null) {//这段逻辑貌似没用了@陆泽,检查下这段逻辑如果没用就干掉
                if (!dataScourceShell.equals(dataSourcesManager.get(id))){//数据源存在且与当前数据源不一致，则给id加时间戳
                    id = id + System.currentTimeMillis();
                    dataScourceShell = new DataScourceShell(this.client, id, id, dataSource);
                    dataSourcesManager.register(dataScourceShell);
                }
                return dataScourceShell;
            } else {
                try {
                    dataSourcesManager.register(dataScourceShell);
                } catch (ZKException e) {
                    e.printStackTrace();
                    throw new RuntimeException(e);
                }
                return dataScourceShell;
            }

        } else {//如果已经有了,需要检查下数据库连通性,看看是否是数据库密码被修改
            
            return dataScourceShell;
        }
        
    }
    
    private LinkedHashMap<String, RasterLayerBean> getLayerMap(Element root, Map<String, RasterDataSetBean> dataSetMap) throws Exception {
        LinkedHashMap<String, RasterLayerBean> layerMap = new LinkedHashMap<String, RasterLayerBean>();
        @SuppressWarnings("unchecked")
        List<Element> items = root.element("Layers").elements();
        for (Element item : items) {
            RasterLayerBean layer = this.createLayer(item, dataSetMap);
            if(layer != null){
                layerMap.put(layer.getId(), layer);
            }
        }
        
        return layerMap;
    }
    
    private RasterLayerBean createLayer(Element info, Map<String, RasterDataSetBean> dataSetMap) throws Exception {
        //处理基本参数
        String layerId = info.attributeValue("Id");
        
        try {
            RasterLayerBean layer = new RasterLayerBean();
            this.getInfoByStr(info, layer, "Id,BeginLevel,EndLevel");
        
            Map<Integer, RasterLayerLevelBean> allLayerLevel = new HashMap<Integer, RasterLayerLevelBean>();
            for (Integer i = layer.getBeginLevel(); i <= layer.getEndLevel(); i++) {
                RasterLayerLevelBean layerLevel = new RasterLayerLevelBean();
                layerLevel.setLevel(i);
                allLayerLevel.put(i, layerLevel);
            }
    
            //补充LayerLevel使用的dataset,如果有其他公用字段，也需要以同补充
            allLayerLevel = this.addDataSetToLayerLevel(info, allLayerLevel, dataSetMap);
            layer.setLevelMap(allLayerLevel);
            
            return layer;
        }catch(Exception e){
            e.printStackTrace();
            throw new Exception(String.format("图层%s:%s",layerId,e.getMessage()));
            
        }

    }
    
    private Map<Integer, RasterLayerLevelBean> addDataSetToLayerLevel(Element info,
            Map<Integer, RasterLayerLevelBean> allLayerLevel,
            Map<String, RasterDataSetBean> dataSetMap) throws Exception {
        if(info.element("DataSet").elements().size() == 0) {
            String dataSetId = info.element("DataSet").getTextTrim();
            if(dataSetMap.get(dataSetId) == null){
                throw new Exception(String.format("没有找到数据集%s", dataSetId));
            }
            
            allLayerLevel.entrySet().forEach( entry -> {
                entry.getValue().setDataSetId(dataSetId);
            });
        } else {
            @SuppressWarnings("unchecked")
            List<Element> items = info.element("DataSet").elements();
            @SuppressWarnings("unused")
            List<String> fieldsList = null;
            for (Element item : items) {
                // 获得数据源
                String dataSetId = this.getDataSetId(item);
                
                RasterDataSetBean dataSet = dataSetMap.get(dataSetId);
                if (dataSet == null) {
                    throw new Exception(String.format("没有找到数据集%s", dataSetId));
                }               
                Integer begin = this.getInteger(item, "begin");
                if(begin == null){
                    begin = this.getInteger(item, "Begin");
                }
                Integer end = this.getInteger(item,"end");
                if(end == null){
                    end = this.getInteger(item,"End");
                }

                for (Integer level = begin; level <= end; level ++) {
                    RasterLayerLevelBean layerLevel = allLayerLevel.get(level);
                    layerLevel.setDataSetId(dataSetId);
                }

            }
        }
        
        return allLayerLevel;
    }
    
    private String getDataSetId(Element item){
        String dataSetName;
        if(item.elements().isEmpty()) {
            dataSetName = item.getStringValue();
        } else {
            dataSetName = item.elementText("DataSetId");
        }
        return dataSetName;
    }
}
