package com.northpool.service.xmlloader.vectorservice;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.commons.lang3.StringUtils;
import org.dom4j.Element;
import org.xml.sax.SAXException;

import com.northpool.commons.reflect.Bean;
import com.northpool.commons.util.MD5;
import com.northpool.resources.Constants.DATA_SOURCE_TYPE;
import com.northpool.resources.Image;
import com.northpool.resources.datasource.IDataSource;
import com.northpool.resources.datasource.MongodbDataSource;
import com.northpool.resources.datasource.db.DbDataSource;
import com.northpool.resources.datasource.raster.RasterDataSource;
import com.northpool.resources.datatable.operate.IColumn;
import com.northpool.resources.datatable.operate.ITableOperator;
import com.northpool.service.client.Client;
import com.northpool.service.config.data_service.DataServiceBean;
import com.northpool.service.config.data_service.DataServiceShell;
import com.northpool.service.config.data_service.IDataService;
import com.northpool.service.config.data_service.raster.IRasterDataService;
import com.northpool.service.config.data_service.raster.RasterDataServiceBean;
import com.northpool.service.config.data_service.raster.RasterDataServiceShell;
import com.northpool.service.config.data_source.IDataSourceInService;
import com.northpool.service.config.raster_service.dataset.RasterDataSetBean;
import com.northpool.service.config.vector_service.dataset.DataSetBean;
import com.northpool.service.manager.abstractclass.ZKException;
import com.northpool.service.manager.data_service.IDataServiceManager;
import com.northpool.service.manager.data_service.raster.IRasterDataServiceManager;


public abstract class AbstractXmlLoader {
	
	protected Client client;
	
	protected String validateXMLPath;
	
	abstract protected String getValidateXMLPath();
	
	protected AbstractXmlLoader(Client client){
		this.client = client;
	}
	
	/**
	 * 解析数据源
	 * 
	 * @param info
	 * @return
	 */
	protected IDataSource createDataSource(Element info) {
		String type = this.getString(info, "Type");
		String id = info.attributeValue("Id");
		String url = this.getString(info,"Url");
		String user = this.getString(info,"User");
		String password = this.getString(info,"Password");
		if ("oracle".equalsIgnoreCase(type) || "postgres".equalsIgnoreCase(type) || "postgre".equalsIgnoreCase(type) || "postgreSQL".equalsIgnoreCase(type)) {
			
			DbDataSource dbDataSource = new DbDataSource(id, url, user, password, null);
			//this.getInfoByStr(info, dbDataSource, "Url,Id,User,Password");

			if("oracle".equalsIgnoreCase(type)){
				dbDataSource.setDataSourceType(DATA_SOURCE_TYPE.oracle);
			}
			
			if("postgres".equalsIgnoreCase(type) || "postgre".equalsIgnoreCase(type) || "postgreSQL".equalsIgnoreCase(type)){
				dbDataSource.setDataSourceType(DATA_SOURCE_TYPE.postgreSQL);
			}
			
			if("mysql".equalsIgnoreCase(type)){
				dbDataSource.setDataSourceType(DATA_SOURCE_TYPE.mysql);
			}
			
			return dbDataSource;
		}
		if ("mongodb".equalsIgnoreCase(type)) {
			
			MongodbDataSource mongodbDataSource = new MongodbDataSource(null,null,null,null,password,null);
			this.getInfoByStr(info, mongodbDataSource, "Ip,Namespace,Id,User");
			Integer port;
			try{
				port = this.getInteger(info, "Port");
			}catch(Exception e){
				port = null;
			}
			if(port == null){
				try{
					port = this.getInteger(info, "Post");
				}catch(Exception e){
					port = null;
				}
			}
			if(port == null){
				throw new RuntimeException("port必须非空且必须是整数");
			}
			mongodbDataSource.setPort(port);
			
			return mongodbDataSource;
		}
		if("raster".equalsIgnoreCase(type)) {
		    String ip = this.getString(info, "Ip");
		    String path = this.getString(info, "Path");
		    try {
                RasterDataSource rasterDataSource = new RasterDataSource(id, ip, path);
                return rasterDataSource;
            } catch (IOException e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            }
		    
		}
		throw new RuntimeException("现在不支持 " + type + " 的数据库");
	}

	/**
	 * 从XML中去信息
	 * 
	 * @param item
	 * @param t
	 * @param fieldsStr
	 *            字段,已逗号隔开
	 */
	protected <T> void getInfoByStr(Element item, T t, String fieldsStr) {
		String[] fieldArr = fieldsStr.split(",");
		for (String field : fieldArr) {
			String value = null;

			if ("Id".equalsIgnoreCase(field)) {
				value = item.attributeValue("Id");
				if(value == null){
					value = "_tmp";
				}
			} else {
				value = this.getString(item, field);
			}
			if (StringUtils.isNotBlank(value)) {
				value = value.trim();
			}
			
			field = field.substring(0, 1).toLowerCase() + field.substring(1);
			try {
				Bean.setSingleObjectValueByFieldName(t, field, value);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				throw new RuntimeException("处理" + field + "出错");
			}
		}
	}
	
	/**
	 * 自动注册数据服务（如果有，则不注册，只能注册只读服务）
	 * @param info
	 * @param id
	 * @return
	 * @throws Exception
	 */
	protected IDataService autoRegisterDBDataService(Element info,IDataSourceInService dataSourceInService,String id, String name) throws Exception {
		
		String tableName = this.getString(info, "Tablename");
		
		String spatialField = this.getString(info, "ShapeField");
		// 处理字段
		String originFields = this.getString(info, "Fields");
		//过滤条件
		String filter = this.getString(info, "SQLFilter");
		//自动创建只能是readonly的数据服务
		Boolean readonly = true;
		
		String[] fields = originFields.split(",");
	
		DataServiceBean dataService = new DataServiceBean(id,spatialField,dataSourceInService.getId(), tableName, readonly, fields,null);
		dataService.setFilter(filter);
		dataService.setName(name);
		IDataServiceManager dataServiceManager = this.client.getDataServiceManager();
		IDataService dataServiceShell = new DataServiceShell(client,dataService);

		if (dataServiceManager.get(id) != null){
			dataServiceManager.unRegister(id);
		}
		dataServiceShell.start();
		try {
		    dataServiceManager.register(dataServiceShell);
		} catch (  ZKException  e) {
			e.printStackTrace();
			throw new Exception(e);
		}

		return dataServiceShell;

	}
	
	protected IRasterDataService autoRegisterRasterDataService(List<Image> images, String id, String name, boolean isTerrain) throws Exception {
	    RasterDataServiceBean rasterDataServiceBean = new RasterDataServiceBean(id, name, images, isTerrain, null);
	    IRasterDataServiceManager dataServiceManager = this.client.getRasterDataServiceManager();
	    RasterDataServiceShell rasterDataServiceShell = new RasterDataServiceShell(this.client, rasterDataServiceBean);
	    // 若数据服务id已存在，则判断是否为同一个数据服务，若相同，则更新zk中的数据服务状态为启动，若不同，则在id之后增加时间戳
	    if(dataServiceManager.get(id) != null) {
	        if(!rasterDataServiceShell.equals(dataServiceManager.get(id))) {
	            id = StringUtils.joinWith("_", id, System.currentTimeMillis());
	            rasterDataServiceBean.setId(id);
	        } else {
	            dataServiceManager.unRegister(id);
	        }
	    }
        rasterDataServiceShell.start();
        try {
            dataServiceManager.register(rasterDataServiceShell);
        } catch (ZKException e) {
            e.printStackTrace();
            throw new Exception(e);
        }
	    
	    return rasterDataServiceShell;
	}
	
	protected DataSetBean createDataSet(DataServiceBean dataService,Element info){
		
		String idInXML = info.attributeValue("Id");
		
		DataSetBean dataSet = new DataSetBean(idInXML,dataService.getId());
		
		String spatialFilter = this.getString(info, "SpatialFilter");
		if (StringUtils.isNotBlank(spatialFilter)) {
			dataSet.setSpatialFilter(spatialFilter);
		}
		
		String filter = this.getString(info, "SQLFilter");
		if (StringUtils.isNotBlank(filter)) {
			dataSet.setFilter(filter);
		}
		
		//以后加入sort
		
		/*String sortStr = info.elementText("Sort");
		if (sortStr != null) {
			if (!"".equalsIgnoreCase(sortStr)) {
				String[] sortArr = sortStr.split("=");
				if (sortArr.length != 2) {
					throw new RuntimeException("数据集 " + dataSet.getDataServiceId() + " 设置排序字段格式错误 ，格式应为 {field}=asc格式");
				}
				if ((!sortArr[1].equalsIgnoreCase("asc")) && (!sortArr[1].equalsIgnoreCase("desc"))) {
					throw new RuntimeException("数据集 " + dataSet.getDataServiceId() + " 设置排序字段格式错误 ，格式应为 {field}=asc格式");
				}
			
				dataSet.setSortFiled(refSortField);
				dataSet.setSort(sortArr[1]);

			}
		}*/
		
		return dataSet;
		
	}
	
	
	
	private void checkVectorInfo(Element item, Map<String, IDataSourceInService> dataSourceMap) throws Exception {
	    String dataSourceId = this.getString(item, "DataSource");
        IDataSourceInService dataSourceInService = dataSourceMap.get(dataSourceId);
        if (dataSourceInService == null) {
            throw new Exception(StringUtils.join("没有找到名称为 ", dataSourceId, " 的数据源"));
        }
        String idInXML = item.attributeValue("Id");
        String fields = this.getString(item, "Fields");
        String spatialFieldsName = this.getString(item, "ShapeField");
        String tableName = this.getString(item, "Tablename");
        
        //开始检查表是否存在，字段是否正确，空间字段是否正确；
        ITableOperator tableBuilder = null;
        try {
            tableBuilder = dataSourceInService.getTableBuilder(tableName);
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception(String.format("dataset %s ,获取%s表信息失败,错误信息：%s",idInXML,tableName,e.getMessage()));
        }
        String[] fieldsArr = fields.split(",");
        for(String fieldName : fieldsArr){
            if(tableBuilder.getColumnMap().get(fieldName) == null){
                throw new Exception(String.format("dataset %s ,%s表没有找到字段 %s", idInXML,tableName,fieldName));
            }
        }
        IColumn column = tableBuilder.getColumnMap().get(spatialFieldsName);
        if(column == null){
            throw new Exception(String.format("dataset %s ,%s表没有找到字段 %s", idInXML,tableName,spatialFieldsName));
        }
        if(!column.isSpatial()){
            throw new Exception(String.format("dataset %s ,%s表,字段 %s,不是空间字段", idInXML,tableName,spatialFieldsName));
        }
	}
	
	/**
     * 根据XML配置解析矢量数据信息并注册
     * @param item 矢量数据DataSet节点根元素
     * @param dataSourceMap 已注册的数据源Map信息
     * @return
     * @throws Exception
     */
	private DataSetBean getVectorDataSet(Element item, Map<String, IDataSourceInService> dataSourceMap) throws Exception {
	    this.checkVectorInfo(item, dataSourceMap);
	    
	    String idInXML = item.attributeValue("Id");
        String dataSourceId = this.getString(item, "DataSource");
        
        IDataSourceInService dataSourceInService = dataSourceMap.get(dataSourceId);
        
        String fields = this.getString(item, "Fields");
        
        String id = StringUtils.join("autocreate_", idInXML, "@", dataSourceInService.mark(), "_", MD5.getMD5String(fields));
    
        IDataService dataServiceShell = this.autoRegisterDBDataService(item, dataSourceInService,id, idInXML);
        
        DataServiceBean dataService = dataServiceShell.getBean();
        
        return this.createDataSet(dataService, item);
	}
	
	/**
	 * 根据XML配置解析影像数据信息并注册
	 * @param item 影像数据DataSet节点根元素
	 * @param dataSourceMap 已注册的数据源Map信息
	 * @return
	 * @throws Exception
	 */
	private RasterDataSetBean getRasterDataSet(Element item, Map<String, IDataSourceInService> dataSourceMap, boolean isTerrain) throws Exception {
	    String idInXML = item.attributeValue("Id");
	    Element elements = item.element("Images");
        @SuppressWarnings("unchecked")
        List<Element> imageElements = elements.elements();
        
        if(imageElements == null || imageElements.isEmpty()) {
            throw new Exception("影像数据服务中必须包含至少一个影像文件");
        }
        
        List<Image> images = new LinkedList<Image>();
        for(Element element : imageElements) {
            String dataSourceId = this.getString(element, "DataSource");
            IDataSourceInService dataSourceInService = dataSourceMap.get(dataSourceId);
            
            if (dataSourceInService == null) {
                throw new Exception(StringUtils.join("没有找到名称为 ", dataSourceId, " 数据源"));
            }
            
            RasterDataSource rasterDataSource = (RasterDataSource) dataSourceInService.getBean();
            
            String fileName = this.getString(element, "FileName");
            if(!rasterDataSource.hasRasterFile(fileName)) {
                throw new Exception(StringUtils.join("未在数据源 ", dataSourceId, " 中找到影像文件 ", fileName));
            }
            images.add(new Image(rasterDataSource, fileName));
        }
        
        IRasterDataService rasterDataServiceShell = this.autoRegisterRasterDataService(images, idInXML, idInXML, isTerrain);
        RasterDataServiceBean rasterDataServiceBean = rasterDataServiceShell.getBean();
        RasterDataSetBean dataSet = new RasterDataSetBean(idInXML, rasterDataServiceBean.getId());
        
        String spatialFilter = this.getString(item, "SpatialFilter");
        if (StringUtils.isNotBlank(spatialFilter)) {
            dataSet.setSpatialFilter(spatialFilter);
        }
        
        return dataSet;
	}
	
	/**
	 * 根据XML配置解析所有数据集信息并注册
	 * @param root XML中服务信息根元素
	 * @param dataSourceMap 已注册的数据源Map信息
	 * @return
	 * @throws Exception
	 */
	protected Map<String, DataSetBean> getDataSetMap(Element root, Map<String, IDataSourceInService> dataSourceMap) throws Exception {
		Map<String, DataSetBean> dataSetMap = new HashMap<String, DataSetBean>();
		@SuppressWarnings("unchecked")
		List<Element> items = root.element("DataSets").elements();
			
		for (Element item : items) {
		    String type = this.getString(item, "Type");
		    
		    if(StringUtils.isBlank(type) || "vector".equalsIgnoreCase(type)) {
		        DataSetBean dataSet = this.getVectorDataSet(item, dataSourceMap);
	            dataSetMap.put(dataSet.getId(), dataSet);
		    }
			
		}
		return dataSetMap;
	}
	
	/**
     * 根据XML配置解析所有数据集信息并注册
     * @param root XML中服务信息根元素
     * @param dataSourceMap 已注册的数据源Map信息
     * @return
     * @throws Exception
     */
    protected Map<String, RasterDataSetBean> getRasterDataSetMap(Element root, Map<String, IDataSourceInService> dataSourceMap) throws Exception {
        Map<String, RasterDataSetBean> dataSetMap = new HashMap<String, RasterDataSetBean>();
        @SuppressWarnings("unchecked")
        List<Element> items = root.element("DataSets").elements();
            
        for (Element item : items) {
            String type = this.getString(item, "Type");
            
            if(StringUtils.isBlank(type) || "image".equalsIgnoreCase(type)) {
                RasterDataSetBean dataSet = this.getRasterDataSet(item, dataSourceMap, false);
                dataSetMap.put(dataSet.getId(), dataSet);
            } else {
                RasterDataSetBean dataSet = this.getRasterDataSet(item, dataSourceMap, true);
                dataSetMap.put(dataSet.getId(), dataSet);
            }
            
        }
        return dataSetMap;
    }
	
	/*protected IDataService createDataService(Element item,IDataSourceInService dataSourceInService,String id) throws Exception{
		
		DATA_SOURCE_TYPE type = dataSourceInService.getDataSourceType();
		switch(type){
			case oracle:{
				return this.autoRegisterDBDataService(item, (DbDataSource)dataSourceInService.getBean(),id);
			}
			case postgreSQL:{
				return this.autoRegisterDBDataService(item, (DbDataSource)dataSourceInService.getBean(),id);
			}
			case mysql:{
				return this.autoRegisterDBDataService(item, (DbDataSource)dataSourceInService.getBean(),id);
			}
		}
		throw new Exception("目前不支持数据源:" + type.name());
	}*/
	
	
	
	protected String getString(Element info,String elementName){
		String valueStr = info.elementTextTrim(elementName);
		if(StringUtils.isBlank(valueStr)){
			valueStr = info.attributeValue(elementName);
		}
		return valueStr;
		
	}
	protected Integer getInteger(Element info,String elementName){
		String valueStr = this.getString(info, elementName);
		
		if(StringUtils.isBlank(valueStr)){
			return null;
		}
		
		try{
			Integer value = Integer.valueOf(valueStr);
			return value;
		}catch(Exception e){
			throw new RuntimeException("xml中" + elementName + "必须是Integer类型");
		}
	}
	
	protected Double getDouble(Element info,String elementName){
		String valueStr = this.getString(info, elementName);
		
		if(StringUtils.isBlank(valueStr)){
			return null;
		}
		
		try{
			Double value = Double.valueOf(valueStr);
			return value;
		}catch(Exception e){
			throw new RuntimeException("xml中" + elementName + "必须是Double类型");
		}
	}
	
	
	protected Boolean getBoolean(Element info,String elementName){
		String valueStr = this.getString(info, elementName);
		
		if(StringUtils.isBlank(valueStr)){
            return null;
        }
		
		try{
			Boolean value = Boolean.valueOf(valueStr);
			return value;
		}catch(Exception e){
			throw new RuntimeException("xml中" + elementName + "必须是Boolean类型");
		}
	}
	
	private void validateXMLSchema(InputStream inXMl) throws SAXException, IOException{
		SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        Schema schema;
		schema = factory.newSchema(new File(this.getValidateXMLPath()));
		Validator validator = schema.newValidator();
        validator.validate(new StreamSource(inXMl));
	}
	
	
	public void validateXML(InputStream inXMl) throws Exception{
		try {
			this.validateXMLSchema(inXMl);
			inXMl.markSupported();
		} catch (SAXException e1) {
			throw new Exception("xml格式错误:" + e1.toString().replaceAll("org.xml.sax.SAXParseException;", ""));
		} catch (IOException e) {
			e.printStackTrace();
			throw new Exception("没有找到XML校验文件");
		} finally{
			inXMl.close();
		}
	}
	

}
