
package com.northpool.service.dao;

import com.northpool.bean.Idable;
import com.northpool.commons.event.EventContainer;
import com.northpool.commons.event.Listener;
import com.northpool.service.client.Client;
import com.northpool.service.config.IBeanShell;
import com.northpool.service.config.data_service.IDataService;
import com.northpool.service.manager.IMetaDataManager;
import com.northpool.service.manager.abstractclass.ZKException;
import com.northpool.service.message.MetaDataChangeMessage;
import com.northpool.structure.queryhashtable.QueryHashTableHeap;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.IgniteCountDownLatch;
import org.apache.ignite.IgniteMessaging;
import org.apache.ignite.cache.*;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.lang.IgniteBiPredicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.cache.Cache;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;
import java.util.concurrent.*;
import java.util.function.Consumer;
import java.util.function.Function;


public abstract class AbstractIgniteDao<B, T extends Idable<String> & IBeanShell<B>> implements IMetaDataDao<T> {
    Logger logger = LoggerFactory.getLogger(AbstractIgniteDao.class);

    protected Ignite ignite;
    protected IgniteMessaging rmtMsg;

    protected IgniteCache<String, B> cache;

    protected QueryHashTableHeap<String, T> table;
    protected String idFieldName;
    protected Client client;
    protected final Boolean readOnly;
    protected String managerRoot;
    protected Boolean isReady = false;
    protected EventContainer<Listener> eventContainer = new EventContainer<Listener>();

    protected IMetaDataManager<T> manager;

    protected static boolean hasListener = false;

    protected static CompletableFuture<String> listener;

    protected static CompletableFuture<CompletableFuture> listenerIsSet;



    protected static ThreadPoolExecutor pool = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, new ArrayBlockingQueue(100000), new RejectedExecutionHandler() {
        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
            throw new RuntimeException("队列已满");
        }
    });

    public AbstractIgniteDao(String idFieldName, QueryHashTableHeap<String,T> table, Client client, String managerRoot, Boolean readOnly, Ignite ignite){
        this.idFieldName = idFieldName;
        this.table = table;
        this.readOnly = readOnly;
        this.managerRoot = managerRoot;
        this.client = client;
        this.ignite = ignite;
        this.rmtMsg = ignite.message(ignite.cluster().forRemotes());

    }

    public void init(){
        try {
            IgniteCache<String, B> cache = ignite.cache(this.managerRoot);
            if (cache == null) {
                this.cache = this.createCache();
            }else {
                this.cache = cache;
            }
            this.addIgniteMessageListener();
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        this.isReady = true;
    }

    //protected abstract void createCache() throws Exception;

    protected IgniteCache<String, B> createCache() throws Exception {
        CacheConfiguration<String, B> cacheCfg = new CacheConfiguration<>();
        cacheCfg.setName(this.managerRoot);
        cacheCfg.setCacheMode(CacheMode.REPLICATED);
        cacheCfg.setSqlSchema("PUBLIC");
        cacheCfg.setWriteSynchronizationMode(CacheWriteSynchronizationMode.FULL_SYNC);
        Class <B> entityClass = (Class <B>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
        QueryEntity queryEntity = new QueryEntity(String.class, entityClass);
        Field[] declaredFields = entityClass.getDeclaredFields();
        for (Field field : declaredFields) {
            queryEntity.addQueryField(field.getName(), field.getType().getName(), null);
        }

       // queryEntity.setIndexes(Arrays.asList(new QueryIndex(this.idFieldName)));
        queryEntity.setKeyFieldName(this.idFieldName);
        queryEntity.setKeyType(String.class.getName());
        cacheCfg.setQueryEntities(Arrays.asList(queryEntity));
        //cacheCfg.setKeyConfiguration(new CacheKeyConfiguration(String.class.getName(), this.idFieldName));
        IgniteCache<String, B> cache = ignite.createCache(cacheCfg);

        return cache;
    }

    public List<T> findAll() throws Exception{
        Iterator<Cache.Entry<String, B>> iterator = this.cache.iterator();
        List<T> list = new ArrayList<>(this.cache.size());
        while(iterator.hasNext()){
            Cache.Entry<String, B> entry = iterator.next();
            list.add(this.getShell(entry.getValue()));
        }
        return list;
    }


    public T findOne(String id) throws Exception{
        return this.getShell(this.cache.get(id));
    }


    public void insert(T t) throws Exception{
        this.cache.put(t.getId(), t.getBean());
        this.syncWait(t.getId(), MetaDataChangeMessage.MetaDataChangeType.CHILD_ADDED);

    }

    public boolean exists(String id) throws ZKException {
        return this.cache.containsKey(id);
    }

    public void delete(String id) throws Exception{
        this.cache.remove(id);
        this.syncWait(id, MetaDataChangeMessage.MetaDataChangeType.CHILD_REMOVED);
    }

    /**
     * 往ZK更新bean
     * @param t
     * @throws ZKException
     */
    public void update(T t) throws ZKException{
        this.cache.put(t.getId(), t.getBean());
        syncWait(t.getId(), MetaDataChangeMessage.MetaDataChangeType.CHILD_UPDATED);
    }

    private void syncWait(String id, MetaDataChangeMessage.MetaDataChangeType type){
        int remoteNodesSize = ignite.cluster().forServers().nodes().size() - 1;//除了自身以外的全部服务节点
        if(remoteNodesSize > 0){
            MetaDataChangeMessage msg = new MetaDataChangeMessage(type, id, this.path(id) + new Date().getTime());
            this.rmtMsg.send(this.managerRoot, msg);
            final IgniteCountDownLatch latch = ignite.countDownLatch(msg.getCountDownId(), remoteNodesSize, true, true);
            latch.await(10, TimeUnit.SECONDS);
        }


    }

    protected void saveBeanToHashTable(T bean){
        this.table.insert(bean);
    }

    protected void removeBeanToHashTable(String pk){
        this.table.remove(pk);
    }

    protected void updateBeanToHashTable(T bean){
        this.table.update(bean);
    }


    protected abstract T getShell(B bean);
    /**
     * 响应事件
     * @throws Exception
     */
    private void addIgniteMessageListener() throws Exception{
        String listenterPath = this.managerRoot;


        this.rmtMsg.localListen(listenterPath, new IgniteBiPredicate<UUID, MetaDataChangeMessage>() {

            @Override
            public boolean apply(UUID nodeId, MetaDataChangeMessage msg) {
                final IgniteCountDownLatch latch = ignite.countDownLatch(msg.getCountDownId(), 0, true, false);
                if (latch == null){
                    logger.error("同步计数器获取失败，请检查");
                }
                String id = msg.getId();
                String path = path(id);
                B bean;
                MetaDataChangeMessage.MetaDataChangeType type = msg.getType();
                switch (type) {
                    case CHILD_ADDED:
                        bean = cache.get(id);
                        try{
                            T shell = getShell(bean);
                            if (shell instanceof IDataService){
                                IDataService dataService = (IDataService) shell;
                                dataService.getDataSource().getBean().getTable(dataService.getTableName()).reload();
                            }
                            saveBeanToHashTable(shell);
                            triggerEvent(id, type);
                        }catch (Exception e){
                            logger.error(managerRoot + ":" + id + "元数据插入失败", e);
                        }
                        break;
                    case CHILD_REMOVED:
                        try {
                            removeBeanToHashTable(id);
                            triggerEvent(id, type);
                        } catch (Exception e) {
                            logger.error(managerRoot + ":" + id + "元数据删除失败", e);
                        }
                        break;
                    case CHILD_UPDATED:
                        bean = cache.get(id);
                        try {
                            T shell = getShell(bean);
                            if (shell instanceof IDataService){
                                IDataService dataService = (IDataService) shell;
                                dataService.getDataSource().getBean().getTable(dataService.getTableName()).reload();
                            }
                            updateBeanToHashTable(shell);
                            triggerEvent(id, type);
                        } catch (Exception e) {
                            e.printStackTrace();
                            logger.error(managerRoot + ":" + id + "元数据更新失败", e);
                        }
                        break;
                    default:
                        break;
                }
                latch.countDown();
                return true;
            }
        });
    }

    private void triggerEvent(String id, MetaDataChangeMessage.MetaDataChangeType eventType) throws ExecutionException, InterruptedException {
        System.out.println(this.manager == null);
        System.out.println(hasListener);
        if (hasListener && this.manager != null){
            CompletableFuture<String> future = null;
            EventMessage message = new EventMessage();
            try{
                future = getListener();
                listener = null;
                listenerIsSet = null;
                message.setBeanType(this.managerRoot);
                message.setEventType(eventType.name());
                message.setId(id);
                if (eventType != MetaDataChangeMessage.MetaDataChangeType.CHILD_REMOVED){
                    message.setBean(this.manager.getJSON(id));
                }
            }catch (Exception e){
                logger.error("引擎同步触发异常",e);
                future.completeExceptionally(e);
            }
            future.complete(message.toString());
        }
    }




    private String path(String id){
        String path = this.managerRoot + "/" + id;
        return path;
    }



    public static CompletableFuture<String> getListener() throws ExecutionException, InterruptedException {
        if (listener != null){//已有监听，则直接返回
            return listener;
        }else {//没有等有了再返回
            listenerIsSet = new CompletableFuture();
            return listenerIsSet.get();
        }
    }

    public static void setListener(Consumer callback, Function exception) {
        hasListener = true;
        listener = new CompletableFuture<>();
        if(callback != null){
            listener.thenAcceptAsync(callback);//给监听设置回调
        }

        if (exception != null){
            listener.exceptionally(exception);
        }
        if (listenerIsSet != null){//有人等待监听设置，则告诉他已经有了监听
            listenerIsSet.complete(listener);
        }
    }
}
