
package com.northpool.service.dao;

import com.northpool.bean.Idable;
import com.northpool.bean.Jsonable;
import com.northpool.bean.JsonableBuilder;
import com.northpool.commons.event.EventContainer;
import com.northpool.commons.event.Listener;
import com.northpool.resources.exception.IdExistsException;
import com.northpool.service.client.Client;
import com.northpool.service.config.IDocumentAble;
import com.northpool.service.config.IVersionAble;
import com.northpool.service.config.data_service.IDataService;
import com.northpool.service.manager.IMetaDataManager;
import com.northpool.service.manager.abstractclass.Json2BeanFailedException;
import com.northpool.service.manager.abstractclass.ZKException;
import com.northpool.structure.queryhashtable.QueryHashTableHeap;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.transaction.CuratorOp;
import org.apache.curator.framework.api.transaction.TransactionOp;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.zookeeper.CreateMode;
import org.bson.BsonDocument;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import static org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent.Type.CHILD_REMOVED;

public abstract class AbstractZkDao <T extends Jsonable & Idable<String> & IDocumentAble,Builder extends JsonableBuilder<T>> implements IMetaDataDao<T> {
    Logger logger = LoggerFactory.getLogger(AbstractZkDao.class);

    protected QueryHashTableHeap<String,T> table;
    protected String idFieldName;
    protected Client client;
    protected final Boolean readOnly;
    protected String managerRoot;
    protected PathChildrenCache HolderCache;
    protected static final String INFO_FIELD = "version";
    protected Builder beanBuilder;
    protected Boolean isReady = false;
    protected EventContainer<Listener> eventContainer = new EventContainer<Listener>();

    protected IMetaDataManager<T> manager;

    protected IMongoDao<T> mongoDao;

    protected static boolean hasListener = false;

    protected static CompletableFuture<String> listener;

    protected static CompletableFuture<CompletableFuture> listenerIsSet;


    protected static ThreadPoolExecutor pool = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, new ArrayBlockingQueue(100000), new RejectedExecutionHandler() {
        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
            throw new RuntimeException("队列已满");
        }
    });

    public AbstractZkDao(String idFieldName,Builder beanBuilder, QueryHashTableHeap<String,T> table, Client client,String managerRoot,Boolean readOnly){
        this.idFieldName = idFieldName;
        this.table = table;
        this.readOnly = readOnly;
        this.managerRoot = client.ROOT + "/" + managerRoot;
        this.client = client;
        this.beanBuilder = beanBuilder;
        //this.mongoDao = mongoDao;
    }

    public void init() throws Exception {
        try {
            this.createRoot();
            this.syncToZK();
            this.addZKListener();
            //this.syncFromZK();
        } catch (Exception e) {
            throw e;
        }
        this.isReady = true;
    }

    public List<T> findAll(){
        return this.mongoDao.find(new BsonDocument());
    }

    public T findOne(String id){
        return this.mongoDao.findone(id);
    }

    private void createRoot() throws Exception{
        if(client.getZoo().checkExists().forPath(this.managerRoot) == null){
            client.getZoo().create().creatingParentsIfNeeded().withMode(CreateMode.PERSISTENT).forPath(this.managerRoot);
        }
    }

    /**
     * hashtable同步到zk
     * @throws Exception
     */
    private synchronized void syncToZK() throws Exception{
        this.isReady = false;
       // client.getZoo().delete().forPath(this.managerRoot);
        this.createRoot();
        List<String> nodes = client.getZoo().getChildren().forPath(this.managerRoot);

        nodes = nodes.stream().map(path -> {
            int index = path.lastIndexOf("/");
            if (index != -1){
                return path.substring(index);
            }else {
                return path;
            }
        }).collect(Collectors.toList());
        List<T> all = this.findAll();
        if (all != null && !all.isEmpty()){
            Iterator<T> iterator = all.iterator();
            while(iterator.hasNext()){
                T t = iterator.next();
                if (nodes.contains(t.getId())){
                    continue;
                }
                this.insert2Zk(t);
            }
        }

        this.isReady = true;
    }

    public void insert(T t) throws Exception{
        this.mongoDao.insertOne(t);
        this.insert2Zk(t);
    }

    protected void insert2Zk(T t) throws Exception{
        TransactionOp transaction = client.getZoo().transactionOp();
        //	CuratorOp createOp1 = transaction.create().withMode(CreateMode.PERSISTENT).forPath(path,id.getBytes());
        //	CuratorOp createOp2 = transaction.create().forPath(path + "/id",t.getId().getBytes());
        //	CuratorOp createOp3 = transaction.create().forPath(path + "/data",t.toJson().getBytes());
        //CuratorTransactionFinal finalTransaction = transaction.create().forPath(path + "/" + t.getId(),t.toJson().getBytes()).and();
        List<CuratorOp> iList = this.createBeanNode(transaction, t);

        client.getZoo().transaction().forOperations(iList);
    }
    /**
     * zk同步到 hashtable
     * @param id
     * @return
     * @throws Exception
     */
    private String getVersionFromZK(String id) throws Exception{
        String path = this.path(id);
        String data_path = path + "/" + INFO_FIELD;
        //	String info = null;
        byte[] info_byte;
        try{
            info_byte = client.getZoo().getData().forPath(data_path);
        }catch(Exception e){
            return null;
          /*  e.printStackTrace();
            throw new Exception("节点" + data_path + " 没有信息");*/
        }
        return new String(info_byte, "utf-8");
    }


    private T getFromMongo(String id) throws Exception {
        try {
            T bean = this.mongoDao.findone(id);
            return bean;
        } catch (Exception e) {
            logger.info(managerRoot + "/" + id + "反序列化失败");
            if (id.startsWith("temp_")) {
                this.delete(id);
            }
            throw new Json2BeanFailedException(e);
        }
    }

    protected void saveBeanToHashTable(T bean){
        this.table.insert(bean);
    }

    protected void removeBeanToHashTable(String pk){
        this.table.remove(pk);
    }

    protected void updateBeanToHashTable(T bean){
        this.table.update(bean);
    }

    /**
     * 响应事件
     * @throws Exception
     */
    private void addZKListener() throws Exception{
        String listenterPath = this.managerRoot;
        this.HolderCache = new PathChildrenCache(client.getZoo(), listenterPath, true);

        this.HolderCache.getListenable().addListener((CuratorFramework client, PathChildrenCacheEvent event) -> {
            if (event.getType() == PathChildrenCacheEvent.Type.CONNECTION_SUSPENDED
            || event.getType() == PathChildrenCacheEvent.Type.CONNECTION_RECONNECTED
            || event.getType() == PathChildrenCacheEvent.Type.CONNECTION_LOST){//除了增、删、改以外的事件忽略
                return;
            }

            ChildData data = event.getData();
            String path = new String(data.getData(), "utf-8");
            int index = path.lastIndexOf("/");
            String id = path;
            if (index != -1){
                id = path.substring(index);
            }
            T bean;
            CompletableFuture future = null;
            switch (event.getType()) {
                case CHILD_ADDED:
                    bean = this.getFromMongo(id);
                    while (bean == null){//mongo和zk可能有延时，逻辑上add之后mongo一定会有数据，因此反复查询获取
                        logger.warn(this.getClass().getName() + ":从mongo再次获取id为" + id + "的数据");
                        bean = this.getFromMongo(id);
                        Thread.sleep(100);
                    }
                    try{
                        if (this.table.get(id) == null){//本机插入时，此事件也会响应，造成插入两次的问题，因此先判断是否存在
                            if (bean instanceof IDataService){
                                IDataService dataService = (IDataService) bean;
                                dataService.getDataSource().getBean().getTable(dataService.getTableName()).reload();
                            }
                            this.saveBeanToHashTable(bean);
                        }
                    }catch (Exception e){
                        e.printStackTrace();
                        if (e.getCause() != null && e.getCause() instanceof IdExistsException){//本机插入时，此事件也会响应，造成插入两次的问题，因此此处忽略此异常

                        }else {
                            throw e;
                        }
                    }
                    this.triggerEvent(id, event.getType());
                    break;
                case CHILD_REMOVED:
                    boolean isExists = this.exists(id);
                    if (!isExists){
                        this.removeBeanToHashTable(id);
                    }else {//如果发现zk server还有这个节点，则认为是更新，需要判断版本是否一致
                        String version = this.getVersionFromZK(id);
                        T cacheBean = this.table.get(id);
                        if (cacheBean instanceof IVersionAble){
                            IVersionAble service = (IVersionAble) cacheBean;
                            if (ObjectUtils.notEqual(service.getVersion(), version)){
                                this.removeBeanToHashTable(id);
                            }
                        }/*else if (cacheBean != null && !zkJsonStr.equals(cacheBean.toJson())){//缓存与zk不一致，则需要删除缓存
                            this.removeBeanToHashTable(id);
                        }*/
                    }
                    this.triggerEvent(path, event.getType());
                    break;
                /*case CHILD_UPDATED:
                    bean = this.getFromMongo(id);
                    this.updateBeanToHashTable(bean);

                    this.triggerEvent(path, event.getType());

                    break;*/
                default:
                    break;
            }
        }, pool);
        this.HolderCache.start(PathChildrenCache.StartMode.BUILD_INITIAL_CACHE);
        //this.HolderCache.start();
    }

    private void triggerEvent(String id, PathChildrenCacheEvent.Type eventType) throws ExecutionException, InterruptedException {
        if (hasListener && this.manager != null){
            CompletableFuture<String> future = null;
            EventMessage message = new EventMessage();
            try{
                future = getListener();
                listener = null;
                listenerIsSet = null;
                message.setBeanType(this.managerRoot);
                message.setEventType(eventType.name());
                message.setId(id);
                if (eventType != CHILD_REMOVED){
                    message.setBean(this.manager.getJSON(id));
                }
            }catch (Exception e){
                e.printStackTrace();
                future.completeExceptionally(e);
            }
            future.complete(message.toString());
        }
    }


    public boolean exists(String id) throws ZKException {
        boolean isExists = false;   
        try {
            isExists = !(client.getZoo().checkExists().forPath(this.path(id)) == null);
        } catch (Exception e) {
            throw new ZKException(e);
        }
        return isExists;
    }

    protected String path(String id){
        String path = this.managerRoot + "/" + id;
        return path;
    }

    private void isReady() throws ZKException{
        if(!this.isReady){
            throw new ZKException("ZK客户端没有准备好");
        }
    }

    /**
     * 初始化节点
     * @param transaction
     * @param t
     * @return
     * @throws Exception
     */
    protected List<CuratorOp> createBeanNode(TransactionOp transaction, T t) throws Exception{
        String id = t.getId();
        String path = this.path(id);
        List<CuratorOp> iList = new ArrayList<CuratorOp>();
        CuratorOp createOp1 = transaction.create().withMode(CreateMode.PERSISTENT).forPath(path,id.getBytes("utf-8"));
        iList.add(createOp1);
        CuratorOp createOp2 = transaction.create().forPath(path + "/id",t.getId().getBytes("utf-8"));
        iList.add(createOp2);

        if (t instanceof IVersionAble){
            String version = ((IVersionAble) t).getVersion();
            if (version == null){
                version = new Date().getTime() + "";
            }
            CuratorOp createOp3 = transaction.create().forPath(path + "/" + INFO_FIELD, version.getBytes("utf-8"));
            iList.add(createOp3);
        }

        this.AddExtendsInfo2BeanNode(path, iList, transaction, t);
        return iList;
    }
    /**
     * 删除节点
     * @param transaction
     * @param id
     * @return
     * @throws Exception
     */
    protected List<CuratorOp> deleteBeanNode(TransactionOp transaction,String id) throws Exception{
        String path = this.path(id);
        List<CuratorOp> iList = new ArrayList<CuratorOp>();
        List<String> nodes = client.getZoo().getChildren().forPath(path);
        for(String node : nodes){
            String sub_path = path + "/" + node;
            CuratorOp delete_sub_node = transaction.delete().forPath(sub_path);
            iList.add(delete_sub_node);
        }
        CuratorOp delete = transaction.delete().forPath(path);
        iList.add(delete);
        return iList;
    }


    /**
     * 注入zk时加入的额外属性抽象类
     * @param path
     * @param iList
     * @param transaction
     * @param t
     * @throws Exception
     */
    protected abstract void AddExtendsInfo2BeanNode(String path,List<CuratorOp> iList,TransactionOp transaction,T t) throws Exception;


    public void delete(String id) throws Exception{
        this.mongoDao.deleteone(id);
        removeFromZk(id);
    }

    private void removeFromZk(String id) throws Exception {
        TransactionOp transaction = client.getZoo().transactionOp();
        List<CuratorOp> iList = this.deleteBeanNode(transaction, id);
        client.getZoo().transaction().forOperations(iList);
    }

    /**
     * 往ZK更新bean
     * @param t
     * @throws ZKException
     */
    public void update(T t) throws Exception{
        String id = t.getId();
        String path = this.path(id);
        try {
            this.mongoDao.updateOne(id, t);
            if(client.getZoo().checkExists().forPath(path) == null){
                this.insert2Zk(t);//mongo有而zk没有，则插入zk
            }else{
                //更新zk
                update2Zk(t, id);
            }
        } catch (Exception e) {
            // TODO Auto-generated catch block
            throw e;
        }
    }

    protected void update2Zk(T t, String id) throws Exception {
        TransactionOp transaction = client.getZoo().transactionOp();
        List<CuratorOp> iListUpdate = new ArrayList<CuratorOp>();
        List<CuratorOp> iListDelete = this.deleteBeanNode(transaction, id);
        iListUpdate.addAll(iListDelete);
        List<CuratorOp> iListAdd = this.createBeanNode(transaction, t);
        iListUpdate.addAll(iListAdd);
        //CuratorTransactionFinal finalTransaction = transaction.create().forPath(path + "/" + t.getId(),t.toJson().getBytes()).and();
        client.getZoo().transaction().forOperations(iListUpdate);
    }

    public static CompletableFuture<String> getListener() throws ExecutionException, InterruptedException {
        if (listener != null){//已有监听，则直接返回
            return listener;
        }else {//没有等有了再返回
            listenerIsSet = new CompletableFuture();
            return listenerIsSet.get();
        }
    }

    public static void setListener(Consumer callback, Function exception) {
        hasListener = true;
        listener = new CompletableFuture<>();
        if(callback != null){
            listener.thenAcceptAsync(callback);//给监听设置回调
        }

        if (exception != null){
            listener.exceptionally(exception);
        }
        if (listenerIsSet != null){//有人等待监听设置，则告诉他已经有了监听
            listenerIsSet.complete(listener);
        }
    }

   /* public static void setListener(Consumer consumer){
        setListener(consumer, null);
    }

    public static void consume(String aa){
        System.out.println(aa);
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
                setListener((bb)->{
                    consume(bb.toString());
                });
            }
        };
        new Thread(runnable).start();

    }
    static {
        setListener((aa)->{
            consume(aa.toString());
        });
    }*/
}
