/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.checker;

import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.hdfs.server.datanode.checker.AsyncChecker;
import org.apache.hadoop.hdfs.server.datanode.checker.ThrottledAsyncChecker;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class StorageLocationChecker {
    public static final Logger LOG = LoggerFactory.getLogger(StorageLocationChecker.class);
    private final AsyncChecker<StorageLocation.CheckContext, VolumeCheckResult> delegateChecker;
    private final Timer timer;
    private final long maxAllowedTimeForCheckMs;
    private final FsPermission expectedPermission;
    private final int maxVolumeFailuresTolerated;

    public StorageLocationChecker(Configuration conf, Timer timer) throws DiskChecker.DiskErrorException {
        this.maxAllowedTimeForCheckMs = conf.getTimeDuration("dfs.datanode.disk.check.timeout", "10m", TimeUnit.MILLISECONDS);
        if (this.maxAllowedTimeForCheckMs <= 0L) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.disk.check.timeout - " + this.maxAllowedTimeForCheckMs + " (should be > 0)");
        }
        this.expectedPermission = new FsPermission(conf.get("dfs.datanode.data.dir.perm", "700"));
        this.maxVolumeFailuresTolerated = conf.getInt("dfs.datanode.failed.volumes.tolerated", 0);
        if (this.maxVolumeFailuresTolerated < -1) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.failed.volumes.tolerated - " + this.maxVolumeFailuresTolerated + " " + "should be greater than or equal to -1");
        }
        this.timer = timer;
        this.delegateChecker = new ThrottledAsyncChecker<StorageLocation.CheckContext, VolumeCheckResult>(timer, conf.getTimeDuration("dfs.datanode.disk.check.min.gap", "15m", TimeUnit.MILLISECONDS), 0L, Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("StorageLocationChecker thread %d").setDaemon(true).build()));
    }

    public List<StorageLocation> check(Configuration conf, Collection<StorageLocation> dataDirs) throws InterruptedException, IOException {
        LinkedHashMap<StorageLocation, Boolean> goodLocations = new LinkedHashMap<StorageLocation, Boolean>();
        HashSet<StorageLocation> failedLocations = new HashSet<StorageLocation>();
        HashMap futures = Maps.newHashMap();
        LocalFileSystem localFS = FileSystem.getLocal((Configuration)conf);
        StorageLocation.CheckContext context = new StorageLocation.CheckContext(localFS, this.expectedPermission);
        for (StorageLocation location : dataDirs) {
            goodLocations.put(location, true);
            Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(location, context);
            if (!olf.isPresent()) continue;
            futures.put(location, olf.get());
        }
        if (this.maxVolumeFailuresTolerated >= dataDirs.size()) {
            throw new DiskChecker.DiskErrorException("Invalid value configured for dfs.datanode.failed.volumes.tolerated - " + this.maxVolumeFailuresTolerated + ". Value configured is >= to the number of configured volumes (" + dataDirs.size() + ").");
        }
        long checkStartTimeMs = this.timer.monotonicNow();
        for (Map.Entry entry : futures.entrySet()) {
            long waitSoFarMs = this.timer.monotonicNow() - checkStartTimeMs;
            long timeLeftMs = Math.max(0L, this.maxAllowedTimeForCheckMs - waitSoFarMs);
            StorageLocation location = (StorageLocation)entry.getKey();
            try {
                VolumeCheckResult result = (VolumeCheckResult)((Object)((ListenableFuture)entry.getValue()).get(timeLeftMs, TimeUnit.MILLISECONDS));
                switch (result) {
                    case HEALTHY: {
                        break;
                    }
                    case DEGRADED: {
                        LOG.warn("StorageLocation {} appears to be degraded.", (Object)location);
                        break;
                    }
                    case FAILED: {
                        LOG.warn("StorageLocation {} detected as failed.", (Object)location);
                        failedLocations.add(location);
                        goodLocations.remove(location);
                        break;
                    }
                    default: {
                        LOG.error("Unexpected health check result {} for StorageLocation {}", (Object)result, (Object)location);
                        break;
                    }
                }
            }
            catch (ExecutionException | TimeoutException e) {
                LOG.warn("Exception checking StorageLocation " + location, e.getCause());
                failedLocations.add(location);
                goodLocations.remove(location);
            }
        }
        if (this.maxVolumeFailuresTolerated == -1 ? dataDirs.size() == failedLocations.size() : failedLocations.size() > this.maxVolumeFailuresTolerated) {
            throw new DiskChecker.DiskErrorException("Too many failed volumes - current valid volumes: " + goodLocations.size() + ", volumes configured: " + dataDirs.size() + ", volumes failed: " + failedLocations.size() + ", volume failures tolerated: " + this.maxVolumeFailuresTolerated);
        }
        if (goodLocations.size() == 0) {
            throw new DiskChecker.DiskErrorException("All directories in dfs.datanode.data.dir are invalid: " + failedLocations);
        }
        return new ArrayList<StorageLocation>(((HashMap)goodLocations).keySet());
    }

    public void shutdownAndWait(int gracePeriod, TimeUnit timeUnit) {
        try {
            this.delegateChecker.shutdownAndWait(gracePeriod, timeUnit);
        }
        catch (InterruptedException e) {
            LOG.warn("StorageLocationChecker interrupted during shutdown.");
            Thread.currentThread().interrupt();
        }
    }
}

