/*
 * Copyright 2016 Azavea
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package geotrellis.raster.resample

import geotrellis.raster._
import geotrellis.vector.Extent


abstract class AggregateResample(tile: Tile, extent: Extent, targetCS: CellSize) extends Resample(tile, extent) {

  val srcCellWidth = extent.width / tile.cols
  val srcCellHeight = extent.height / tile.rows
  val halfSrcCellWidth = srcCellWidth / 2
  val halfSrcCellHeight = srcCellHeight / 2

  val halfTargetWidth = targetCS.width / 2
  val halfTargetHeight = targetCS.height / 2

  def xIndices(x: Double): (Int, Int) = {
    // Distance from the left of tile
    val dLeftX = x - extent.xmin - halfSrcCellWidth

    // calc cell distance from left edge
    val dLeftCellLeft: Double = dLeftX - halfTargetWidth
    val dLeftCellRight: Double = dLeftX + halfTargetWidth

    // Calculate indices
    val leftIndex =
      if (dLeftCellLeft > 0) (dLeftCellLeft / srcCellWidth).ceil.toInt
      else 0
    val rightIndex =
      if(dLeftCellRight < extent.width) (dLeftCellRight / srcCellWidth).floor.toInt
      else tile.cols - 1

    (leftIndex, rightIndex)
  }
  def yIndices(y: Double): (Int, Int) = {
    // Distance from top of tile
    val dTopY = extent.ymax - y - halfSrcCellHeight

    // calc cell distance from top
    val dTopCellTop: Double = dTopY - halfTargetHeight
    val dTopCellBottom: Double = dTopY + halfTargetHeight

    // Calculate indices
    val topIndex =
      if (dTopCellTop > 0) (dTopCellTop / srcCellHeight).ceil.toInt
      else 0
    val bottomIndex =
      if (dTopCellBottom < extent.height) (dTopCellBottom / srcCellHeight).floor.toInt
      else tile.rows - 1

    (topIndex, bottomIndex)
  }

  def contributions(x: Double, y: Double): Seq[(Int, Int)] = {
    val (xLow, xHigh) = xIndices(x)
    val (yLow, yHigh) = yIndices(y)

    if (xLow > xHigh || yLow > yHigh) Vector[(Int, Int)]()
    else for {
      xs <- xLow to xHigh
      ys <- yLow to yHigh
    } yield (xs, ys)
  }
}
