package com.geoway.atlas.data.vector.spark.common.rpc.client;

import com.geoway.atlas.data.vector.spark.common.rpc.*;
import com.geoway.atlas.data.vector.spark.common.rpc.common.GrpcExceptionUtil;
import com.google.protobuf.Empty;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author zhaotong 2022/9/9 18:02
 */
public class SparkRpcClientApi {

    public final static int MAX_INBOUND_METADATA_SIZE = 1024 * 1024 * 10;

    private static final Logger logger = LoggerFactory.getLogger(SparkRpcClientApi.class);

    private final ManagedChannel channel;
    private final SparkDataManagerGrpc.SparkDataManagerBlockingStub blockDataStub;
    //    private final SparkVectorDataManagerGrpc.SparkVectorDataManagerFutureStub futureDataStub;
    private final SparkDataManagerGrpc.SparkDataManagerStub dataStub;
    private final SparkVectorProcessGrpc.SparkVectorProcessBlockingStub blockProcessStub;
    private final SparkVectorProcessGrpc.SparkVectorProcessStub processStub;

    private final AtlasBaseOpGrpc.AtlasBaseOpBlockingStub blockAtlasBaseOpStub;
    private final AtlasBaseOpGrpc.AtlasBaseOpStub atlasBaseOpStub;

    public SparkRpcClientApi(String target, ExecutorService rpcClientPool) {
        this(ManagedChannelBuilder.forTarget(target)
                .maxInboundMetadataSize(MAX_INBOUND_METADATA_SIZE), rpcClientPool);
    }

    public SparkRpcClientApi(String address, int port, ExecutorService rpcClientPool) {

        this(ManagedChannelBuilder.forAddress(address, port)
                .maxInboundMetadataSize(MAX_INBOUND_METADATA_SIZE), rpcClientPool);
    }

    public SparkRpcClientApi(ManagedChannelBuilder<?> channelBuilder, ExecutorService rpcClientPool) {
        logger.info("启动客户端");
        this.channel = channelBuilder.executor(rpcClientPool).usePlaintext().build();
        this.blockDataStub = SparkDataManagerGrpc.newBlockingStub(this.channel);
        this.dataStub = SparkDataManagerGrpc.newStub(this.channel);
//        this.futureDataStub = SparkVectorDataManagerGrpc.newStub(this.channel);
        this.blockProcessStub = SparkVectorProcessGrpc.newBlockingStub(this.channel);
        this.processStub = SparkVectorProcessGrpc.newStub(this.channel);

        this.blockAtlasBaseOpStub = AtlasBaseOpGrpc.newBlockingStub(this.channel);
        this.atlasBaseOpStub = AtlasBaseOpGrpc.newStub(this.channel);
    }

    /**
     * 载入数据
     *
     * @param dataType        数据类型
     * @param atlasRpcDataTag 矢量数据标签
     * @param dataStoreFormat 载入数据类型
     * @param loadParams      载入数据参数
     * @param taskId          任务id
     * @param jobId           工作项id
     * @return 返回载入信息
     */
    public String loadData(String dataType,
                           AtlasRpcDataTag atlasRpcDataTag,
                           String dataStoreFormat,
                           Map<String, String> loadParams,
                           String taskId,
                           String jobId) {
        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        dataStub.loadData(
                RpcLoadDataParams
                        .newBuilder()
                        .setDataType(dataType)
                        .setDataTag(atlasRpcDataTag)
                        .setDataStoreFormat(dataStoreFormat)
                        .putAllLoadParams(loadParams)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }


    /**
     * 保存矢量数据
     *
     * @param dataType        数据类型
     * @param atlasRpcDataTag 矢量数据标签
     * @param dataStoreFormat 载入数据类型
     * @param saveParams      载入数据参数
     * @param taskId          任务id
     * @param jobId           工作项id
     * @return 返回保存信息
     */
    public String saveData(String dataType,
                           AtlasRpcDataTag atlasRpcDataTag,
                           String dataStoreFormat,
                           Map<String, String> saveParams,
                           String taskId,
                           String jobId) {
        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.saveData(
                RpcSaveDataParams
                        .newBuilder()
                        .setDataType(dataType)
                        .setDataTag(atlasRpcDataTag)
                        .setDataStoreFormat(dataStoreFormat)
                        .putAllSaveParams(saveParams)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }

    /**
     * 展示数据名称和数据标签
     *
     * @return 返回数据名称与数据标签
     */
    public List<AtlasRpcDataTag> showData() {

        final RpcShowDataRespond[] result = new RpcShowDataRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcShowDataRespond> streamObserver = new StreamObserver<RpcShowDataRespond>() {
            @Override
            public void onNext(RpcShowDataRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.showData(Empty.newBuilder().build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getDataTagsList();
    }

    /**
     * 删除数据
     *
     * @param atlasRpcDataTag 数据标签
     *                        其中如果atlasdataname=AtlasRpcDataName("","")则删除所有匹配的label
     * @param taskId          任务id
     * @param jobId           jobid
     * @return 返回消息
     */
    public String deleteData(AtlasRpcDataTag atlasRpcDataTag,
                             String taskId,
                             String jobId) {

        final RpcDeleteRespond[] result = new RpcDeleteRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcDeleteRespond> streamObserver = new StreamObserver<RpcDeleteRespond>() {
            @Override
            public void onNext(RpcDeleteRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.deleteData(RpcDeleteDataParams
                .newBuilder()
                .setDeleteDataTag(atlasRpcDataTag)
                .setTaskId(taskId)
                .setJobId(jobId)
                .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getMessage();
    }

    /**
     * 展示数据名称和数据标签
     *
     * @return 返回数据名称与数据标签
     */
    public RpcDescDataRespond descData(
            AtlasRpcDataTag resultRpcDataTag, String taskId, String jobId) {

        final RpcDescDataRespond[] result = new RpcDescDataRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcDescDataRespond> streamObserver = new StreamObserver<RpcDescDataRespond>() {
            @Override
            public void onNext(RpcDescDataRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.descData(
                RpcDescDataParams.newBuilder()
                        .setDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0];
    }

    /**
     * 判断是否包含指定数据
     *
     * @return 返回是否包含指定的数据
     */
    public RpcExistDataRespond existData(
            AtlasRpcDataTag resultRpcDataTag, String taskId, String jobId) {

        final RpcExistDataRespond[] result = new RpcExistDataRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcExistDataRespond> streamObserver = new StreamObserver<RpcExistDataRespond>() {
            @Override
            public void onNext(RpcExistDataRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.existData(
                RpcExistDataParams.newBuilder()
                        .setDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0];
    }

    /**
     * 判断是否包含指定数据
     *
     * @return 返回是否包含指定的数据
     */
    public RpcPersistRespond persistData(
            AtlasRpcDataTag resultRpcDataTag, String taskId, String jobId) {

        final RpcPersistRespond[] result = new RpcPersistRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcPersistRespond> streamObserver = new StreamObserver<RpcPersistRespond>() {
            @Override
            public void onNext(RpcPersistRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.persistData(
                RpcPersistDataParams.newBuilder()
                        .setPersistDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0];
    }

    /**
     * 判断是否包含指定数据
     *
     * @return 返回是否包含指定的数据
     */
    public RpcUnPersistRespond unPersistData(
            AtlasRpcDataTag resultRpcDataTag, String taskId, String jobId) {

        final RpcUnPersistRespond[] result = new RpcUnPersistRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcUnPersistRespond> streamObserver = new StreamObserver<RpcUnPersistRespond>() {
            @Override
            public void onNext(RpcUnPersistRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.unPersistData(
                RpcUnPersistDataParams.newBuilder()
                        .setUnpersistDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0];
    }

    /**
     * 判断是否包含指定数据
     *
     * @return 返回是否包含指定的数据
     */
    public RpcRenameDataRespond renameData(
            AtlasRpcDataTag rawRpcDataTag,
            AtlasRpcDataTag newRpcDataTag,
            String taskId, String jobId) {

        final RpcRenameDataRespond[] result = new RpcRenameDataRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRenameDataRespond> streamObserver = new StreamObserver<RpcRenameDataRespond>() {
            @Override
            public void onNext(RpcRenameDataRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.dataStub.renameData(
                RpcRenameDataParams.newBuilder()
                        .setRawDataTag(rawRpcDataTag)
                        .setNewDataTag(newRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0];
    }

    /**
     * 无输入源执行器
     *
     * @param processName      执行器名称
     * @param processParams    执行器执行参数
     * @param resultRpcDataTag 结果数据标签
     * @return 返回执行信息
     */
    public String nilProcess(String processName,
                             Map<String, String> processParams,
                             AtlasRpcDataTag resultRpcDataTag,
                             String taskId,
                             String jobId) {

        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.processStub.nilProcess(
                RpcNilProcessParams
                        .newBuilder()
                        .setProcess(processName)
                        .putAllProcessParams(processParams)
                        .setResultDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }


    /**
     * 单输入源执行器
     *
     * @param atlasRpcDataTag  矢量数据标签
     * @param processName      执行器名称
     * @param processParams    执行器执行参数
     * @param resultRpcDataTag 结果数据描述
     * @return 返回执行信息
     */
    public String unitaryProcess(AtlasRpcDataTag atlasRpcDataTag,
                                 String processName,
                                 Map<String, String> processParams,
                                 AtlasRpcDataTag resultRpcDataTag,
                                 String taskId,
                                 String jobId) {

        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.processStub.unitaryProcess(
                RpcUnitaryProcessParams
                        .newBuilder()
                        .setDataTag(atlasRpcDataTag)
                        .setProcess(processName)
                        .putAllProcessParams(processParams)
                        .setResultDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }


    /**
     * 双输入源执行器
     *
     * @param baseDatTag        基础矢量图层标签
     * @param otherDatTag       其他矢量图层标签
     * @param processName     执行器名称
     * @param processParams   执行器执行参数
     * @param resultDatTag      结果数据标签
     * @return 返回执行信息
     */
    public String binaryProcess(AtlasRpcDataTag baseDatTag,
                                AtlasRpcDataTag otherDatTag,
                                String processName,
                                Map<String, String> processParams,
                                AtlasRpcDataTag resultDatTag,
                                String taskId,
                                String jobId) {
        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.processStub.binaryProcess(
                RpcBinaryProcessParams
                        .newBuilder()
                        .setBaseDataTag(baseDatTag)
                        .setOtherDataTag(otherDatTag)
                        .setProcess(processName)
                        .putAllProcessParams(processParams)
                        .setResultDataTag(resultDatTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }

    /**
     * 多输入源执行器
     *
     * @param atlasRpcDataTags 矢量数据标签
     * @param processName      执行器名称
     * @param processParams    执行器执行参数
     * @param resultRpcDataTag 结果数据描述
     * @return 返回执行信息
     */
    public String listProcess(List<AtlasRpcDataTag> atlasRpcDataTags,
                              String processName,
                              Map<String, String> processParams,
                              AtlasRpcDataTag resultRpcDataTag,
                              String taskId,
                              String jobId) {

        final RpcRespondMessage[] result = new RpcRespondMessage[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcRespondMessage> streamObserver = new StreamObserver<RpcRespondMessage>() {
            @Override
            public void onNext(RpcRespondMessage value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.processStub.listProcess(
                RpcListProcessParams
                        .newBuilder()
                        .addAllDataTags(atlasRpcDataTags)
                        .setProcess(processName)
                        .putAllProcessParams(processParams)
                        .setResultDataTag(resultRpcDataTag)
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getResponse();
    }

    public void startMonitorTask(Map<String, String> envParams) {
        RpcAtlasBaseOpRespons rpcAtlasBaseOpRespons =
                this.blockAtlasBaseOpStub.startMonitorTask(EnvInfo.newBuilder().putAllEnvParams(envParams).build());
        logger.info(rpcAtlasBaseOpRespons.getResponse());
    }

    public String cancelTask(String taskId) {
        final RpcTaskRespond[] result = new RpcTaskRespond[1];
        final Throwable[] throwables = new Throwable[1];
        CountDownLatch countDownLatch = new CountDownLatch(1);

        StreamObserver<RpcTaskRespond> streamObserver = new StreamObserver<RpcTaskRespond>() {
            @Override
            public void onNext(RpcTaskRespond value) {
                result[0] = value;
            }

            @Override
            public void onError(Throwable t) {
                throwables[0] = t;
                countDownLatch.countDown();
            }

            @Override
            public void onCompleted() {
                countDownLatch.countDown();
            }
        };

        this.processStub.cancelTask(
                RpcTaskId.newBuilder()
                        .setTaskId(taskId)
                        .build(), streamObserver);

        try {
            countDownLatch.await();
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (throwables[0] != null) {
            GrpcExceptionUtil.throwException(throwables[0]);
        }

        return result[0].getRespond();
    }

    public String startJob(String taskId, String jobId, String responseInfo) {
        return this.blockAtlasBaseOpStub.startJob(
                AtlasJobInfo.newBuilder()
                        .setTaskId(taskId)
                        .setJobId(jobId)
                        .setMessage(responseInfo).build()).getResponse();
    }

    public void finishJob(String taskId, String jobId, String responseInfo) {
        this.blockAtlasBaseOpStub.finishJob(
                AtlasJobInfo.newBuilder().setTaskId(taskId).setJobId(jobId).setMessage(responseInfo).build());
    }

    public void cancelJob(String taskId, String jobId, String responseInfo) {
        this.blockAtlasBaseOpStub.cancelJob(
                AtlasJobInfo.newBuilder().setTaskId(taskId).setJobId(jobId).setMessage(responseInfo).build());
    }

    public String getJobReponse(String taskId, String jobId) {
        return this.blockAtlasBaseOpStub.getResponse(
                AtlasJobInfo.newBuilder().setTaskId(taskId).setJobId(jobId).build()).getResponse();
    }


    public void stop() throws InterruptedException {
        if (channel != null) {
            channel.shutdown().awaitTermination(3, TimeUnit.SECONDS);
        }
    }
}
