package com.geoway.atlas.web.api.v2.service.pkg.impl;

import com.geoway.atlas.common.error.NotSupportException;
import com.geoway.atlas.common.utils.UUIDUtils;
import com.geoway.atlas.data.common.data.AtlasDataName;
import com.geoway.atlas.data.vector.spark.common.rpc.AtlasRpcDataTag;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcDescDataRespond;
import com.geoway.atlas.web.api.v2.component.rpc.RpcClientProxy;
import com.geoway.atlas.data.vector.spark.common.rpc.common.Constants;
import com.geoway.atlas.web.api.v2.domain.metadata.LayerMetadata;
import com.geoway.atlas.web.api.v2.dto.pack.CreateOidLayerInfo;
import com.geoway.atlas.web.api.v2.exception.AtlasException;
import com.geoway.atlas.web.api.v2.service.DataServer;
import com.geoway.atlas.web.api.v2.service.ProcessServer;
import com.geoway.atlas.web.api.v2.service.pkg.SpatialStatisticServer;
import com.geoway.atlas.web.api.v2.utils.AtlasTagUtils;
import com.geoway.atlas.web.api.v2.utils.InputParamUtils;
import com.geoway.atlas.web.api.v2.utils.ResponseBuilder;
import com.geoway.atlas.web.api.v2.utils.SecurityUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author zhaotong 2022/11/7 10:30
 */
@Slf4j
@Service
public class SpatialStatisticServerImpl implements SpatialStatisticServer {

    @Autowired
    private DataServer dataServer;

    @Autowired
    private ProcessServer processServer;

    @Autowired
    private RpcClientProxy client;

    @Autowired
    private Constants constants;

    @Override
    public Map<String, Object> calculateProcess(
            String baseIdentity,
            Map<String, String> calculateAreaParams,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId,
            String jobId) {

        String processName = constants.calculateProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(calculateAreaParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                calculateAreaParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合面积计算流程!");
    }

    @Override
    public Map<String, Object> groupByProcess(
            String baseIdentity,
            String filterSql,
            Map<String, String> groupByParams,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId,
            String jobId) {

        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(filterSql), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String filterLayerName = filter(baseLayerName, filterSql, null, taskId, jobId);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(filterLayerName),
                constants.statisticProcessName(),
                groupByParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成分组统计流程!");
    }


    @Override
    public Map<String, Object> byDictProcess(String baseIdentity,
                                             String filterSql,
                                             Map<String, String> byColParams,
                                             String resultIdentity,
                                             boolean persistDisk,
                                             String resultUrl,
                                             String taskId,
                                             String jobId) {
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(filterSql), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String filterLayerName = filter(baseLayerName, filterSql, null, taskId, jobId);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(filterLayerName),
                constants.statisticByColumnProcessName(),
                byColParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成分组统计流程!");
    }

    @Override
    public Map<String, Object> byShapeProcess(String baseIdentity,
                                              String filterSql,
                                              String staticIdentity,
                                              String rowPropertyField,
                                              String valueField,
                                              Map<String, String> byShapeParams,
                                              String resultIdentity,
                                              boolean persistDisk,
                                              String resultUrl,
                                              String taskId,
                                              String jobId) {
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(filterSql), taskId, UUIDUtils.getUUID());

        // 获取统计图层的相关信息
        String staticLayerName = dataServer.loadVectorLayer(staticIdentity, new HashMap<>(), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        // 获取当前图层的图形字段信息
        String baseLayerShapeName = getShapeField(baseLayerName, taskId);

        // 面积计算
        String nAreaName = "area_" + Hex.encodeHexString(RandomUtils.nextBytes(4));
        String calAreaLayer = calArea(baseLayerName, taskId, nAreaName);

        // 获取筛选后的基础图层
        String filterBaseLayerName = filter(calAreaLayer, filterSql,
                (baseLayerShapeName + "," +
                        rowPropertyField.trim() + "," +
                        valueField.trim() + "," +
                        nAreaName).toLowerCase(), taskId, UUIDUtils.getUUID());

        // 获取随机的后缀
        String suffix = Hex.encodeHexString(RandomUtils.nextBytes(2));
        String oidName = "oid_" + Hex.encodeHexString(RandomUtils.nextBytes(4));

        CreateOidLayerInfo createOidLayerInfo = createOid(staticLayerName, taskId, oidName);

        String overlayOidName = createOidLayerInfo.getOidName() + "_" + suffix;
        // 持久化图层
        client.getSparkRpcClientApi()
                .persistData(AtlasTagUtils.getAtlasRpcDataTag(createOidLayerInfo.getLayerName()), taskId, UUIDUtils.getUUID());
        log.info("完成" + createOidLayerInfo.getLayerName() + "数据持久化!");
        // 叠加计算
        String overlayName = overlayNoRepair(
                createOidLayerInfo.getLayerName(),
                suffix,
                filterBaseLayerName,
                null,
                taskId,
                UUIDUtils.getUUID());

        // 重算面积字段
        String reCalAreaLayer = reCalArea(overlayName, nAreaName, valueField.trim(), taskId);

        String nStaticLayerNM = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_st_" + UUIDUtils.getUUID();
        // 按列统计
        byShapeParams.put(constants.staticByColPropertyField(), overlayOidName);
        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(reCalAreaLayer),
                constants.statisticByColumnProcessName(),
                byShapeParams,
                AtlasTagUtils.getAtlasRpcDataTag(nStaticLayerNM),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

        // 合并到原始图层
        // atlas.process.field.left.base.fields
        // atlas.process.field.right.base.fields
        // atlas.process.field.join.fields
        String nJoinLayerNM = (createOidLayerInfo.isDelete()) ? AtlasDataName.TEMP_LAYER_PREFIX() + "_o_j_" + UUIDUtils.getUUID() : resultLayerName;
        Map<String, String> fieldJoinParams = new HashMap<>();
        fieldJoinParams.put(constants.fieldJoinLeftBaseFields(), createOidLayerInfo.getOidName());
        fieldJoinParams.put(constants.fieldJoinRightBaseFields(), overlayOidName);

        String message2 =
                client.getSparkRpcClientApi().binaryProcess(
                        AtlasTagUtils.getAtlasRpcDataTag(createOidLayerInfo.getLayerName()),
                        AtlasTagUtils.getAtlasRpcDataTag(nStaticLayerNM),
                        constants.fieldJoinProcessName(),
                        fieldJoinParams,
                        AtlasTagUtils.getAtlasRpcDataTag(nJoinLayerNM),
                        taskId,
                        UUIDUtils.getUUID());
        log.info(message2);

        // 如果是临时创建的唯一标识字段则需要删除
        if (createOidLayerInfo.isDelete()) {
            Map<String, String> fieldRemoveParams = new HashMap<>();
            fieldRemoveParams.put(constants.fieldRemoveProcessRmFields(), createOidLayerInfo.getOidName());
            String message3 = client.getSparkRpcClientApi().unitaryProcess(
                    AtlasTagUtils.getAtlasRpcDataTag(nJoinLayerNM),
                    constants.fieldRemoveProcessName(),
                    fieldRemoveParams,
                    AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                    taskId,
                    UUIDUtils.getUUID());
            log.info(message3);
        }

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成分组统计流程!");
    }

    @Override
    public Map<String, Object> gridStatistic(String baseIdentity,
                                             boolean saveShape,
                                             Map<String, String> geosotParams,
                                             String resultIdentity,
                                             boolean persistDisk,
                                             String resultUrl,
                                             String taskId,
                                             String jobId) {

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(geosotParams), taskId, UUIDUtils.getUUID());
        String resultLayerName =
                dataServer.generalResultLayerName(resultIdentity);

        String message;
        // 获取基础图层的信息
        RpcDescDataRespond baseLayerDescResponse = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(baseLayerName), taskId, UUIDUtils.getUUID());

        String geoField = baseLayerDescResponse.getDataDescMap().get(constants.dataDescDefaultGeometry());
        String geoEnv = baseLayerDescResponse.getDataDescMap().get(constants.dataDescBoundary());

        // 1. 过滤出参与后续计算的字段
        String filterLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_grid_filter_" + RandomStringUtils.randomAlphanumeric(6);

        Map<String, String> filterMap = new HashMap<>();
        filterMap.put(constants.filterSelectFields(),
                geoField + "," + geosotParams.get(constants.staticByColRowProperty()) + "," +
                        geosotParams.get(constants.gridSplitRecomputeFields()));
        message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                constants.filterProcessName(),
                filterMap,
                AtlasTagUtils.getAtlasRpcDataTag(filterLayerName),
                taskId, UUIDUtils.getUUID());
        log.info("网格统计(1/5):" + message);

        // 2. 进行网格切分
        String splitLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_grid_split_" + RandomStringUtils.randomAlphanumeric(6);
        message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(filterLayerName),
                constants.gridSplitProcessName(),
                geosotParams,
                AtlasTagUtils.getAtlasRpcDataTag(splitLayerName),
                taskId, UUIDUtils.getUUID());
        log.info("网格统计(2/5):" + message);

        String splitStaticLayerName;
        if (saveShape) {
            splitStaticLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_grid_bycol" + RandomStringUtils.randomAlphanumeric(6);
        } else {
            splitStaticLayerName = resultLayerName;
        }

        message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(splitLayerName),
                constants.statisticByColumnProcessName(),
                geosotParams,
                AtlasTagUtils.getAtlasRpcDataTag(splitStaticLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info("网格统计(3/5):" + message);

        if (saveShape) {
            Map<String, String> geoGeneratorParams = new HashMap<>();
            String geomSql;
            if (Integer.parseInt(geosotParams.get(constants.gridSplitMode())) == constants.gridSplitDistinct()) {
                geomSql = "ST_GeomFromGwGrid(" + geosotParams.get(constants.gridSplitCodeField()) + "," +
                        geosotParams.get(constants.gridSplitCodeParam()) + ")";
            } else {
                geomSql = "ST_GeomFromGeoSot(" + geosotParams.get(constants.gridSplitCodeField()) + ")";
            }
            geoGeneratorParams.put(constants.geometryGeneratorSqlExpression(), geomSql);
            if(StringUtils.isNotBlank(geoField)) {
                geoGeneratorParams.put(constants.geometryGeneratorGeomName(), geoField);
                geoGeneratorParams.put(constants.geometryGeneratorGeomSrid(),
                        baseLayerDescResponse.getDataDescMap().get(constants.dataDescCrs()));
                if (StringUtils.isNotBlank(geoEnv)) {
                    geoGeneratorParams.put(constants.geometryGeneratorGeomBoundary(), geoEnv);
                }
            }
            message = client.getSparkRpcClientApi().unitaryProcess(
                    AtlasTagUtils.getAtlasRpcDataTag(splitStaticLayerName),
                    constants.geometryGeneratorProcessName(),
                    geoGeneratorParams,
                    AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                    taskId,
                    UUIDUtils.getUUID());
        } else {
            message = "跳过解码步骤!";
        }
        log.info("Geosot网格统计(4/5):" + message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
            log.info("Geosot网格统计(5/5): 完成数据输出到成果库!");
        } else {
            log.info("Geosot网格统计(5/5): 跳过数据输出!");
        }

        return ResponseBuilder.buildSuccess("完成Geosot网格化流程!");
    }

    @Override
    public Map<String, Object> sqlProcess(
            String baseIdentity, String baseSuffix,
            String otherIdentity, String otherSuffix,
            Map<String, String> paramMaps,
            String resultIdentity, boolean persistDisk, String persistUrl,
            String nTaskId, String jobId) {
        // 获取图层1名称相关信息
        if (StringUtils.isNotBlank(baseIdentity)) {
            if (baseIdentity.contains(":") && StringUtils.startsWithAny(baseIdentity.toLowerCase(), constants.getAllFormats())) {
                throw new AtlasException("不支持数据源直接作为SQL输入源，必须通过算子生成结果图层名称后才能接入！");
            }
        }
        String layerName1 = dataServer.loadVectorLayer(baseIdentity, new HashMap<>(), nTaskId, UUIDUtils.getUUID());
        AtlasRpcDataTag rpcDataTag1 = AtlasTagUtils.getAtlasRpcDataTag(layerName1);

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        AtlasRpcDataTag rpcDataTag2 = null;
        if (StringUtils.isNotBlank(otherIdentity)) {
            if (otherIdentity.contains(":") && StringUtils.startsWithAny(otherIdentity.toLowerCase(), constants.getAllFormats())) {
                throw new AtlasException("不支持数据源直接作为SQL输入源，必须通过算子生成结果图层名称后才能接入！");
            }
            String layerName2 = dataServer.loadVectorLayer(otherIdentity, new HashMap<>(), nTaskId, UUIDUtils.getUUID());
            rpcDataTag2 = AtlasTagUtils.getAtlasRpcDataTag(layerName2);
        }

        if (StringUtils.isNotBlank(baseSuffix) || StringUtils.isNotBlank(otherSuffix)) {
            StringBuilder sb = new StringBuilder();
            if (StringUtils.isNotBlank(baseSuffix)) {
                sb.append(baseSuffix.trim());
            }
            if (rpcDataTag2 != null) {
                sb.append(",");
                if (StringUtils.isNotBlank(otherSuffix)) {
                    sb.append(otherSuffix);
                }
            }
            paramMaps.put(constants.sqlTableExtentSuffix(), sb.toString());
        }

        /**
         * 解码sql语句
         */
        if (paramMaps.containsKey(constants.runSqlExpression())) {
            paramMaps.put(constants.runSqlExpression(),
                    SecurityUtils.dencrypt(paramMaps.get(constants.runSqlExpression())));
        }

        List<AtlasRpcDataTag> atlasRpcDataTags = new ArrayList<>();
        atlasRpcDataTags.add(rpcDataTag1);
        if (rpcDataTag2 != null) {
            atlasRpcDataTags.add(rpcDataTag2);
        }

        String message = client.getSparkRpcClientApi().listProcess(
                atlasRpcDataTags,
                constants.runSqlProcessName(),
                paramMaps,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                nTaskId,
                UUIDUtils.getUUID());

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, persistUrl, nTaskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成SQL查询流程!");
    }


    @Override
    public Map<String, Object> balanceDiffProcess(
            String baseIdentity,
            Map<String, String> balanceDiffParams,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId,
            String jobId) {

        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, new HashMap<>(), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                constants.statisticBalanceDiffName(),
                balanceDiffParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成平差流程!");
    }

    /**
     * 属性筛选
     *
     * @param layerName 图形名称
     * @param filterSql 筛选表达式
     * @param taskId    任务id
     * @param jobId     jobId
     * @return 返回筛选后的图层
     */
    private String filter(String layerName, String filterSql, String filterFields, String taskId, String jobId) {
        if (StringUtils.isNotBlank(filterSql) || StringUtils.isNotBlank(filterFields)) {
            String tempLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_f_" + UUIDUtils.getUUID();
            Map<String, String> filterParams = new HashMap<>();
            if (StringUtils.isNotBlank(filterSql)) {
                filterParams.put(constants.filterExpression(), filterSql);
            }
            if (StringUtils.isNotBlank(filterFields)) {
                filterParams.put(constants.filterSelectFields(), filterFields);
            }
            String message =
                    processServer.filterProcess(layerName, filterParams, tempLayerName, taskId, jobId);
            log.info(message);
            return tempLayerName;
        } else {
            return layerName;
        }
    }


    /**
     * 获取图形字段名称
     *
     * @param layerName 图层名称
     * @param taskId    任务Id
     * @return 返回图形字段名称
     */
    private String getShapeField(String layerName, String taskId) {
        // 获取当前图层的字段信息
        RpcDescDataRespond rpcDescDataRespond = client.getSparkRpcClientApi().descData(
                AtlasTagUtils.getAtlasRpcDataTag(layerName),
                taskId, UUIDUtils.getUUID());
        String baseLayerShapeName = rpcDescDataRespond.getDataDescMap().getOrDefault(constants.dataDescDefaultGeometry(), null);
        if (StringUtils.isBlank(baseLayerShapeName)) {
            throw new NotSupportException("不支持非空间图层进行叠加赋值操作", Thread.currentThread(), 3);
        }
        return baseLayerShapeName;
    }


    private CreateOidLayerInfo createOid(String layerName,
                                         String taskId,
                                         String nOidName) {

        Map<String, String> descMap = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(layerName), taskId, UUIDUtils.getUUID())
                .getDataDescMap();

        boolean delete = false;
        String oidFields = "";
        String resultLayer = layerName;
        if (descMap.containsKey(constants.dataDescOidFields())) {
            oidFields = descMap.get(constants.dataDescOidFields());
            if (oidFields.contains(",")) {
                log.warn("不支持图层包含多个oid,需要重建oid!");
                delete = true;
            }
        } else {
            delete = true;
        }
        if (delete) {
            oidFields = nOidName;
            resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_id_" + UUIDUtils.getUUID();
            Map<String, String> createOidParams = new HashMap<>();
            createOidParams.put(constants.createOidFieldName(), oidFields);
            createOidParams.put(constants.createOidLongField(), "true");
            client.getSparkRpcClientApi().unitaryProcess(AtlasTagUtils.getAtlasRpcDataTag(layerName),
                    constants.createOidProcessName(),
                    createOidParams,
                    AtlasTagUtils.getAtlasRpcDataTag(resultLayer),
                    taskId,
                    UUIDUtils.getUUID());
        }

        return new CreateOidLayerInfo(delete, oidFields, resultLayer);
    }

    private String calArea(String layerName, String taskId, String areaName) {
        String jobId = UUIDUtils.getUUID();
        String resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_a_" + jobId;
        Map<String, String> calAreaParams = new HashMap<>();
        calAreaParams.put(constants.calculateTarget(), constants.calculateTargetStArea());
        calAreaParams.put(constants.calculateFieldName(), areaName);
        processServer.calculatorProcess(layerName, calAreaParams, resultLayer, taskId, jobId);
        return resultLayer;
    }

    private String reCalArea(String layerName, String rawArea, String reCalAreaField, String taskId) {
        String shapeField = getShapeField(layerName, taskId);
        String jobId = UUIDUtils.getUUID();
        String resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_ra_" + jobId;
        Map<String, String> calAreaParams = new HashMap<>();
        String calSql = "st_area(" + shapeField + ") / " + rawArea + " * " + reCalAreaField;
        calAreaParams.put(constants.fieldCalculatorExpression(), calSql);
        calAreaParams.put(constants.fieldCalculatorFieldName(), reCalAreaField);
        calAreaParams.put(constants.fieldCalculatorIsNew(), "false");
        processServer.unitaryProcess(layerName, constants.fieldCalculatorProcessName(), calAreaParams, resultLayer, taskId, jobId);
        return resultLayer;
    }

    private String overlayNoRepair(String baseLayerName, String baseSuffix, String otherLayerName, String otherSuffix, String taskId, String jobId) {
        // 获取执行结果
        String preResultLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_r_" + jobId;
        Map<String, String> processParams = new HashMap<>();
        processParams.put(constants.intersectionIsRepair(), "false");
        if (StringUtils.isNotBlank(baseSuffix)) {
            processParams.put(constants.intersectionLeftSuffix(), baseSuffix);
        }
        if (StringUtils.isNotBlank(otherSuffix)) {
            processParams.put(constants.intersectionRightSuffix(), otherSuffix);
        }
        processServer.binaryProcess(baseLayerName,
                otherLayerName, constants.intersectionProcessName(),
                processParams, preResultLayerName, taskId, jobId);
        log.info("叠加执行结果图层为:" + preResultLayerName);
        return preResultLayerName;
    }
}
