package com.geoway.atlas.web.api.v2.service.pkg.impl.assigin;

import com.geoway.atlas.function.parser.common.FunctionPlan;
import com.geoway.atlas.function.parser.common.QualifiedName;
import com.geoway.atlas.web.api.v2.exception.AtlasException;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author zhaotong 2024/12/20 9:25
 */
@Slf4j
public abstract class AssignFunctionPlan implements FunctionPlan {

    public static final String CURRENT_LAYER = "#0";

    public static final String LEFT_LAYER = "#1";

    public static final String RIGHT_LAYER = "#2";

    // g1a2f3d7作为固定赋值图层后缀主要是为了能够避免基础图层字段冲突，可以保证叠加后的基础图层字段名称不变，赋值图层统一加上此后缀
    public static final String FIXED_ASSIGN_SUFFIX = "g1a2f3d7";

    public static final String FIXED_ASSIGN_LAYER_SUFFIX = "7d3f2a1g";

    private static final Pattern FIELD_PATTERN = Pattern.compile("((#[0-9]+)\\s*\\.\\s*(`.*?`|\\w+))");

    protected String RATE_NAME = "rate_" + AssignFunctionPlan.FIXED_ASSIGN_LAYER_SUFFIX;

    /**
     * 重算面积计算字段
     */
    protected List<QualifiedName> computeFields = new ArrayList<>();

    /**
     * 求交后ShapeArea字段名称
     */
    @Setter
    protected String intersectShapeAreaName = null;

    /**
     * 基础图层ShapeArea名字
     */
    @Setter
    @Getter
    protected String baseShapeAreaName = null;

    @Setter
    protected Map<String, String> defaultValueMap;

    @Setter
    protected boolean isConsiderNull;

    /**
     * 重算字段名称
     *
     * @return 返回重算字段列表
     */
    public List<QualifiedName> getReComputeFields(){
        return computeFields;
    }

    /**
     * 判断是否需要计算叠加图斑的shapearea
     * @return 返回是否需要计算
     */
    public boolean isNeedShapeArea() {
        return !isNeedRecomputeArea();
    }

    /**
     * 判断当前方法是否需要重算面积
     */
    public boolean isNeedRecomputeArea() {
        return !getReComputeFields().isEmpty();
    }

    /**
     * 获取左图层重算面积字段名称
     * @return 返回所有左图层重算面积字段名称
     */
    public List<String> getLeftRecomputeFields(){
        if(!computeFields.isEmpty()){
            return computeFields.stream()
                    .filter(qn -> StringUtils.equals(qn.getFirstName(), LEFT_LAYER))
                    .map(QualifiedName::getSimpleLastName)
                    .collect(Collectors.toList());
        }else return new ArrayList<>();
    }

    /**
     * 获取右图层重算面积字段名称
     * @return 返回所有右图层重算面积字段名称
     */
    public List<String> getRightRecomputeFields(){
        if(!computeFields.isEmpty()){
            return computeFields.stream()
                    .filter(qn -> StringUtils.equals(qn.getFirstName(), RIGHT_LAYER))
                    .map(QualifiedName::getSimpleLastName)
                    .collect(Collectors.toList());
        }else return new ArrayList<>();
    }

    /**
     * 判断是否进行求交融合
     */
    public abstract boolean needRepair();

    /**
     * 获取当前方法不参与计算的字段名称，以便删除字段，减少表的内存占用
     * @param fields 叠加计算后全部字段名称
     * @param assignFieldMap 赋值图层字段映射
     * @return 返回不参与当前计算的字段名称
     */
    public List<String> unusedFields(List<String> fields,
                                              Map<String, String> assignFieldMap){
        List<String> result = new ArrayList<>(fields);
        result.removeAll(getAssignFieldsWithSuffix(assignFieldMap.keySet()));
        if(isNeedRecomputeArea()){
            if(!getLeftRecomputeFields().isEmpty()){
                result.removeAll(getLeftRecomputeFields());
            }

            if(!getRightRecomputeFields().isEmpty()){
                result.removeAll(getAssignFieldsWithSuffix(getRightRecomputeFields()));
            }
        }

        if(isNeedShapeArea()){
            result.remove(baseShapeAreaName);
        }
        return result;
    }

    /**
     * 组合sql的第一步，获取基础统计表
     * @param middleLayer 中间图层名称
     * @param assignFields 赋值字段名称
     * @param oidField 主键名称
     * @return 返回结果
     */
    public String getStatisticSql(String middleLayer, Collection<String> assignFields, String oidField) {

        List<String> groupFields = new ArrayList<>();
        groupFields.add(oidField);
        groupFields.addAll(getAssignFieldsWithSuffix(assignFields));

        List<String> areaFields = new ArrayList<>();
        List<String> anyValueFields = new ArrayList<>();

        if(isNeedRecomputeArea()) {
            areaFields.addAll(getLeftRecomputeFields());
            areaFields.addAll(getAssignFieldsWithSuffix(getRightRecomputeFields()));
        }

        if(isNeedShapeArea()){
            // shapeareaname应该总存在
            if(StringUtils.isBlank(intersectShapeAreaName)){
                throw new AtlasException("没有找到ShapeArea字段!");
            }
            areaFields.add(intersectShapeAreaName);
            anyValueFields.add(baseShapeAreaName);
        }

        String groupFieldSql = groupFields.stream().map(s -> "`" + s + "`").collect(Collectors.joining(","));
        String areaFieldSql = ", " + areaFields.stream().map(s -> "sum(" + s + ") as " + s).collect(Collectors.joining(","));
        String anyValueFieldSql = (anyValueFields.isEmpty()) ? "" : "," + anyValueFields.stream().map(s -> "first_value(" + s + ") as " + s).collect(Collectors.joining(","));

        return String.format("select %s %s %s from %s group by %s", groupFieldSql, areaFieldSql, anyValueFieldSql, middleLayer, groupFieldSql);
    }

    /**
     * 组合sql的第二步，获取赋值的sql
     * @param statisticSql 统计sql
     * @param oidField oid字段的名称
     * @param assignFieldMaps 赋值字段映射，采用linkedhashmap，保证了keyset的顺序
     * @result 返回结果
     */
    public String getAssignFuncSql(String statisticSql, String oidField, Map<String, String> assignFieldMaps){
        String statisticViewAlias = "a" +"_" + FIXED_ASSIGN_LAYER_SUFFIX;
        String selectSql = getAssignSelectSql(statisticViewAlias, assignFieldMaps);
        String groupByKey = String.format("%s.%s", statisticViewAlias, oidField);
        return String.format("select %s.%s, %s from (%s) %s group by %s", statisticViewAlias, oidField, selectSql, statisticSql, statisticViewAlias, groupByKey);
    }

    /**
     * 执行赋值的select语句
     * @param statisticViewAlias 统计表的别名
     * @param assignFieldMaps 赋值字段映射
     * @return 返回select的sql语句
     */
    protected abstract String getAssignSelectSql(String statisticViewAlias, Map<String, String> assignFieldMaps);

    /**
     * 获取添加后缀的赋值图层字段名称
     * @param fieldNames 原始赋值图层字段
     * @return 返回添加后缀后的字段名称
     */
    protected List<String> getAssignFieldsWithSuffix(Collection<String> fieldNames){
        if(fieldNames != null && !fieldNames.isEmpty()) {
            return fieldNames.stream()
                    .map(this::addSuffixName)
                    .collect(Collectors.toList());
        }else return new ArrayList<>();
    }

    protected String addSuffixName(String name){
        return name + "_" + AssignFunctionPlan.FIXED_ASSIGN_SUFFIX;
    }

    /**
     * 替换SQL表达式中的字符串
     * @param sql 原始sql表达式
     * @param layerAlias 基础图层的别名
     * @return 返回替换后的sql表达式
     */
    protected String replaceSqlFields(String sql, String layerAlias){
        Map<String, Function<String, String>> fieldNameMap = new HashMap<>();
        return replaceSqlFields(sql, layerAlias, fieldNameMap);
    }


    /**
     * 替换SQL表达式中的字符串
     * @param sql 原始sql表达式
     * @param layerAlias 基础图层的别名
     * @return 返回替换后的sql表达式
     */
    protected String replaceSqlFields(String sql, String layerAlias, Map<String, Function<String, String>> fieldNameMap){
        StringBuilder sb = new StringBuilder();
        Matcher matcher = FIELD_PATTERN.matcher(sql);
        int retainStart = 0;
        while (matcher.find()){
            if(matcher.start(1) != retainStart){
                sb.append(StringUtils.substring(sql, retainStart, matcher.start(1)));
            }

            String layerIdentity = matcher.group(2);
            String fieldName = matcher.group(3).trim();

            if(StringUtils.equals(LEFT_LAYER, layerIdentity)){
                sb.append(layerAlias).append(".").append(fieldName);
            }

            if(StringUtils.equals(RIGHT_LAYER, layerIdentity)){
                sb.append(layerAlias).append(".");
                if(StringUtils.startsWith(fieldName, "`")){
                    sb.append("`")
                            .append(this.addSuffixName(StringUtils.substring(fieldName, 1, fieldName.length() - 1).trim()))
                            .append("`");
                }else {
                    sb.append(this.addSuffixName(fieldName));
                }
            }

            if(StringUtils.equals(CURRENT_LAYER, layerIdentity)){
                String fieldKey = (fieldName.startsWith("`")) ? fieldName.substring(1, fieldName.length() - 1) : fieldName;
                if(fieldNameMap.containsKey(fieldKey)){
                    sb.append(fieldNameMap.get(fieldKey).apply(layerAlias));
                }else{
                    throw new AtlasException("未找到当前列：" + (layerIdentity + "." + fieldName));
                }
            }

            retainStart = matcher.end(1);
        }
        if(retainStart != sql.length()){
            sb.append(StringUtils.substring(sql, retainStart, sql.length()));
        }
        String nSql = sb.toString();
        log.info(String.format("原始表达式为：%s，替换字段后表达式为：%s", sql, nSql));
        return nSql;
    }

    protected String getFieldNameInLayer(QualifiedName qualifiedName){
        if(StringUtils.equals(LEFT_LAYER, qualifiedName.getFirstName())){
            return qualifiedName.getSimpleLastName();
        }else if(StringUtils.equals(RIGHT_LAYER, qualifiedName.getFirstName())){
            return addSuffixName(qualifiedName.getSimpleLastName());
        }else throw new AtlasException("不支持解析其他图层标识的字段！");
    }

    public String getFieldSelectWithDefExpr(String viewName, Collection<String> valueSet){
        List<String> sqlList = new ArrayList<>();
        for(String fieldName:valueSet){
            if(defaultValueMap.containsKey(fieldName) && StringUtils.isNotBlank(defaultValueMap.get(fieldName))){
                Object defaultValue = defaultValueMap.get(fieldName);
                if(StringUtils.equalsIgnoreCase(defaultValue.toString(), "null")){
                    sqlList.add(String.format("%s.%s", viewName, fieldName));
                }else {
                    sqlList.add(getDefaultValExpr(viewName, fieldName));
                }
            }else {
                sqlList.add(getUndefineValExpr(viewName, fieldName));
            }
        }
        return String.join(", ", sqlList);
    }

    /**
     * 获取默认值表达式
     * @param viewName 视图名称
     * @param fieldName 字段名称
     * @return 返回字段表达式
     */
    protected abstract String getDefaultValExpr(String viewName, String fieldName);

    /**
     * 获取未定义值表达式
     * @param viewName 视图名称
     * @param fieldName 字段名称
     * @return 返回字段表达式
     */
    protected abstract String getUndefineValExpr(String viewName, String fieldName);

}
