package com.geoway.atlas.web.api.v2.utils;

import com.geoway.atlas.common.config.AtlasSystemProperties;
import com.geoway.atlas.web.api.v2.job.JobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

/**
 * @author zhaotong 2024/3/12 15:24
 */
public class PluginParallelManager {

    private static ConcurrentMap<String, ArrayBlockingQueue<Integer>> parallelManagerMap = new ConcurrentHashMap<>();

    private static Logger logger = LoggerFactory.getLogger(PluginParallelManager.class);

    private static final Integer PARALLEL_NUM =
            Integer.parseInt(AtlasSystemProperties.getProperty("atlas.plugin.parallel.num", "1"));

    public static <T> T runPlugin(String pluginName,
                                  Function<Object[], T> pluginFunction,
                                  Object[] pluginParams,
                                  JobManager jobManager,
                                  String jobId,
                                  String taskId) {
        // 检查是否存在当前插件的并发控制队列
        ArrayBlockingQueue<Integer> parallerManager =
                parallelManagerMap.computeIfAbsent(pluginName, s -> new ArrayBlockingQueue<>(PARALLEL_NUM, true));
        // 检查jobid是否还运行
        if (checkPluginIsRunning(jobManager, jobId)) {
            // 将jobid阻塞方式插入到运行队列中
            // 当队列满了阻塞当前线程，直到有出队的任务
            try {
                logger.info("并发控制任务:" + jobId + "准备启动!");
                parallerManager.put(1);
                logger.info("并发控制任务:" + jobId + "已启动!");
            } catch (InterruptedException e) {
                throw new RuntimeException("无法添加到运行中队列");
            }
            // 阻塞线程继续执行时候job可能已经失效，重新监测一下插件是否还在执行
            try {
                if (checkPluginIsRunning(jobManager, jobId)) {
                    return pluginFunction.apply(pluginParams);
                }
            } finally {
                try {
                    logger.info("并发控制任务:" + jobId + "已完成!");
                    parallerManager.poll();
                } catch (Throwable ignored) {
                }
            }
        }

        throw new RuntimeException("任务: " + taskId + ", 工作项: " + jobId + " 已停止!");
    }

    public static boolean checkPluginIsRunning(JobManager jobManager, String jobId) {
        boolean isRunning = false;
        Map<String, Object> jobResponse = jobManager.getJobInfo(jobId);
        if (ResponseBuilder.STATE_RUNNING.equals(jobResponse.get(ResponseBuilder.STATE))) {
            isRunning = true;
        }
        return isRunning;
    }
}
