package com.geoway.atlas.web.api.v2.service.pkg.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.geoway.atlas.common.utils.UUIDUtils;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcRenameDataRespond;
import com.geoway.atlas.web.api.v2.component.rpc.RpcClientProxy;
import com.geoway.atlas.data.vector.spark.common.rpc.common.Constants;
import com.geoway.atlas.web.api.v2.service.DataServer;
import com.geoway.atlas.web.api.v2.service.pkg.DataManagerServer;
import com.geoway.atlas.web.api.v2.utils.AtlasTagUtils;
import com.geoway.atlas.web.api.v2.utils.InputParamUtils;
import com.geoway.atlas.web.api.v2.utils.ResponseBuilder;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

/**
 * @author zhaotong 2022/11/4 0:13
 */
@Slf4j
@Service
public class DataManagerServerImpl implements DataManagerServer {

    @Autowired
    private DataServer dataServer;

    @Autowired
    private RpcClientProxy client;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private Constants constants;

    @Override
    public Map<String, Object> filterProcess(
            String baseIdentity, Map<String, String> filterParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.filterProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(filterParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                filterParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合属性筛选流程!");
    }

    @Override
    public Map<String, Object> vectorIoProcess(
            String baseIdentity, Map<String, String> ioParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        Map<String, String> ioParams2 = new HashMap<>(ioParams);
        ioParams2.putAll(InputParamUtils.getInputVectorParams(ioParams));
        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, ioParams2, taskId, UUIDUtils.getUUID());

        String processName = constants.filterProcessName();

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                ioParams2,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, ioParams2, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成矢量数据转换流程!");
    }

    @Override
    public Map<String, Object> graphIoProcess(
            String baseIdentity, Map<String, String> ioParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String baseLayerName =
                dataServer.loadGraphLayer(baseIdentity, ioParams, taskId, UUIDUtils.getUUID());

        String processName = constants.filterProcessName();

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);
        if (persistDisk) {
            dataServer.saveGraphLayer(baseLayerName, resultUrl, ioParams, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成图数据转换流程!");
    }

    @Override
    public Map<String, Object> toGraphProcess(
            String baseIdentity, Map<String, String> ioParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(ioParams), taskId, UUIDUtils.getUUID());

        String processName = constants.dataToGraphProcessName();

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                ioParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveGraphLayer(resultLayerName, resultUrl, ioParams, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成数据转换流程!");
    }

    @Override
    public Map<String, Object> calcFieldProcess(
            String baseIdentity, Map<String, String> calcFieldParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.fieldCalculatorProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(calcFieldParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                calcFieldParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合字段计算流程!");
    }

    @Override
    public Map<String, Object> geoGeneratorProcess(
            String baseIdentity, Map<String, String> geoGeneratorParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.geometryGeneratorProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(geoGeneratorParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                geoGeneratorParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成图斑生成计算流程!");
    }

    @Override
    public Map<String, Object> overlayRelate(
            String baseIdentity,
            Map<String, String> calcFieldParams,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId,
            String jobId) {
        String processName = constants.overlayRelateProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(calcFieldParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                calcFieldParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合字段计算流程!");
    }

    @Override
    public Map<String, Object> fieldRenameProcess(
            String baseIdentity, Map<String, String> fieldRenameParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.fieldRenameProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(fieldRenameParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                fieldRenameParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合字段重命名流程!");
    }

    @Override
    public Map<String, Object> projectProcess(
            String baseIdentity, Map<String, String> crsParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.transformCrsProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(crsParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                crsParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());
        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合坐标转换流程!");
    }

    public Map<String, Object> mergeProcess(
            String baseIdentity, String mergeIdentity, Map<String, String> mergeParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {
        String layerIdentity = baseIdentity;
        if (StringUtils.isNotBlank(mergeIdentity)) {
            layerIdentity += (";" + mergeIdentity);
        }

        Map<String, String> fullParams = new HashMap<>(InputParamUtils.getInputVectorParams(mergeParams));
        fullParams.putAll(mergeParams);
        String layerName = dataServer.loadVectorLayer(layerIdentity, fullParams, taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        RpcRenameDataRespond renameDataRespond = client.getSparkRpcClientApi().renameData(
                AtlasTagUtils.getAtlasRpcDataTag(layerName),
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }
        return ResponseBuilder.buildSuccess("完成组合合并数据流程!");
    }

    @Override
    public Map<String, Object> fieldJoinProcess(
            String baseIdentity, String otherIdentity,
            Map<String, String> fieldJoinParams,
            String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.fieldJoinProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, new HashMap<>(), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, new HashMap<>(), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        log.info("进入属性连接流程，参数为:" + toJSON(fieldJoinParams));

        String message =
                client.getSparkRpcClientApi().binaryProcess(
                        AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                        AtlasTagUtils.getAtlasRpcDataTag(otherLayerName),
                        processName,
                        fieldJoinParams,
                        AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                        taskId,
                        UUIDUtils.getUUID());

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, UUIDUtils.getUUID());
        }
        return ResponseBuilder.buildSuccess(message);
    }

    @SneakyThrows
    public String toJSON(Object obj) {
        return objectMapper.writeValueAsString(obj);
    }
}
