package com.geoway.atlas.web.api.v2.service.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.geoway.atlas.common.error.NotFoundException;
import com.geoway.atlas.common.error.NotSupportException;
import com.geoway.atlas.common.utils.UUIDUtils;
import com.geoway.atlas.data.common.data.AtlasDataName;
import com.geoway.atlas.data.common.storage.AtlasDataParams;
import com.geoway.atlas.data.common.storage.AtlasDataParams$;
import com.geoway.atlas.data.vector.spark.common.rpc.AtlasRpcDataTag;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcDescDataRespond;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcExistDataRespond;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcRenameDataRespond;
import com.geoway.atlas.data.vector.spark.common.rpc.common.AtlasRpcServerException;
import com.geoway.atlas.web.api.v2.component.rpc.RpcClientProxy;
import com.geoway.atlas.data.vector.spark.common.rpc.common.Constants;
import com.geoway.atlas.web.api.v2.dao.MetadataDao;
import com.geoway.atlas.web.api.v2.domain.metadata.LayerMetadata;
import com.geoway.atlas.web.api.v2.exception.AtlasException;
import com.geoway.atlas.web.api.v2.service.DataServer;
import com.geoway.atlas.web.api.v2.utils.AtlasTagUtils;
import com.geoway.atlas.web.api.v2.utils.ResponseBuilder;
import com.geoway.atlas.web.api.v2.utils.SecurityUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 数据服务实现类
 *
 * @author zhaotong 2022/9/9 16:14
 */
@Slf4j
@Service
public class DataServerImpl implements DataServer {

    @Autowired
    private RpcClientProxy client;

    @Autowired
    private MetadataDao metadataDao;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private Constants constants;

//    @Value("${datasource.jdbc.partition}")
//    private Integer jdbcPartitionNum;


    /**
     * 载入矢量数据
     *
     * @param dataName        数据名称
     * @param dataStoreFormat 数据源格式
     * @param params          载入数据参数
     * @param taskId          taskId作为自动清除后缀
     * @return 返回载入数据后的正常返回值
     */
    @Override
    public Map<String, Object> loadVector(String dataName,
                                          String dataStoreFormat,
                                          Map<String, String> params,
                                          String taskId,
                                          String jobId) {

        String message = client.getSparkRpcClientApi()
                .loadData(constants.dataTypeVector(), AtlasTagUtils.getAtlasRpcDataTag(dataName), dataStoreFormat, params, taskId, jobId);
        return ResponseBuilder.buildSuccess(message);
    }


    /**
     * 输出矢量数据
     *
     * @param dataName        数据名称
     * @param dataStoreFormat 数据源格式
     * @param params          输出数据参数
     * @param taskId          taskId作为自动清除后缀
     * @return 返回载入数据后的正常返回值
     */
    @Override
    public Map<String, Object> saveVector(String dataName,
                                          String dataStoreFormat,
                                          Map<String, String> params,
                                          String taskId,
                                          String jobId) {

        String message = client.getSparkRpcClientApi()
                .saveData(constants.dataTypeVector(), AtlasTagUtils.getAtlasRpcDataTag(dataName), dataStoreFormat, params, taskId, jobId);
        return ResponseBuilder.buildSuccess(message);
    }

    @Override
    public Map<String, Object> showVector() {
        List<AtlasRpcDataTag> rpcDataTags = client.getSparkRpcClientApi().showData();
        return ResponseBuilder.buildSuccess(rpcDataTags);
    }

    @Override
    public Map<String, Object> descVector(String dataName,
                                          String taskId,
                                          String jobId) {
        RpcDescDataRespond respond =
                client.getSparkRpcClientApi().descData(AtlasTagUtils.getAtlasRpcDataTag(dataName), taskId, jobId);
        return ResponseBuilder.buildSuccess(respond);
    }

    @Override
    public Map<String, Object> existVector(String dataName,
                                           String taskId,
                                           String jobId) {
        RpcExistDataRespond respond =
                client.getSparkRpcClientApi().existData(AtlasTagUtils.getAtlasRpcDataTag(dataName), taskId, jobId);
        return ResponseBuilder.buildSuccess(respond);
    }

    @Override
    public Map<String, Object> renameData(String rawDataName,
                                          String newDataName,
                                          String taskId,
                                          String jobId) {
        RpcRenameDataRespond respond =
                client.getSparkRpcClientApi().renameData(
                        AtlasTagUtils.getAtlasRpcDataTag(rawDataName),
                        AtlasTagUtils.getAtlasRpcDataTag(newDataName),
                        taskId,
                        jobId);
        return ResponseBuilder.buildSuccess(respond);
    }


    public String loadVectorLayer(String layerIdentity,
                                  Map<String, String> params,
                                  String taskId,
                                  String jobId) {
        if (StringUtils.isBlank(layerIdentity)) {
            throw new AtlasRpcServerException("载入图层名称为空!");
        }
        // 如果多个图层用分号分隔
        String[] layerSources = layerIdentity.split(";");
        // 如果为一个图层
        if (layerSources.length == 1) {
            String layerSource = layerSources[0];
            return loadSingleLayer(layerSource, params, taskId, jobId);
        } else {
            // 如果结果为多个图层
            // 先生成临时图层名称
            String resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_in_u_" + UUIDUtils.getUUID();
            List<String> resultLayers = new ArrayList<>();
            for (String layerSource : layerSources) {
                resultLayers.add(loadSingleLayer(layerSource, params, taskId, jobId));
            }
            List<AtlasRpcDataTag> rpcDataTags =
                    resultLayers.stream()
                            .map(AtlasTagUtils::getAtlasRpcDataTag)
                            .collect(Collectors.toList());
            // 合并图层
            String message = client.getSparkRpcClientApi().listProcess(
                    rpcDataTags,
                    constants.mergeDataProcessName(),
                    params,
                    AtlasTagUtils.getAtlasRpcDataTag(resultLayer),
                    taskId, jobId);
            log.info(message);
            return resultLayer;
        }

    }


    public String loadGraphLayer(String layerIdentity,
                                 Map<String, String> params,
                                 String taskId,
                                 String jobId) {
        if (StringUtils.isBlank(layerIdentity)) {
            throw new AtlasRpcServerException("载入图层名称为空!");
        }
        // 如果多个图层用分号分隔
        String[] layerSources = layerIdentity.split(";");
        // 如果为一个图层
        if (layerSources.length == 1) {
            String layerSource = layerSources[0];
            return loadSingleLayer(layerSource, params, taskId, jobId);
        } else {
            throw new NotSupportException("不支持同时加载多个图!", Thread.currentThread(), 3);
        }
    }

    public String loadRasterLayer(String layerIdentity,
                                  Map<String, String> params,
                                  String taskId,
                                  String jobId) {
        if (StringUtils.isBlank(layerIdentity)) {
            throw new AtlasRpcServerException("载入图层名称为空!");
        }
        // 如果多个图层用分号分隔
        String[] layerSources = layerIdentity.split(";");
        // 如果为一个图层
        if (layerSources.length == 1) {
            String layerSource = layerSources[0];
            return loadSingleLayer(layerSource, params, taskId, jobId);
        } else {
            throw new NotSupportException("不支持同时加载多个栅格数据!", Thread.currentThread(), 3);
        }
    }

    private String loadSingleLayer(String layerSource, Map<String, String> extraParams, String taskId, String jobId) {
        int maoHaoIndex = layerSource.indexOf(":");
        // 当图层标识中包含冒号
        if (maoHaoIndex != -1) {
            // 获取图层类型
            String format = layerSource.substring(0, maoHaoIndex);
            if (LayerMetadata.SPARK.equalsIgnoreCase(format)) {
                return layerSource.substring(maoHaoIndex + 1);
            }
            if (StringUtils.startsWithIgnoreCase(format, LayerMetadata.SECRET)) {
                int secretType = Integer.parseInt(StringUtils.substring(format, LayerMetadata.SECRET.length()));
                String rawPath = SecurityUtils.dencrypt(StringUtils.substring(layerSource, maoHaoIndex + 1), secretType);
                return loadSingleLayer(rawPath, extraParams, taskId, jobId);
            }

            String dataIoName = constants.getDataIoName(format.toLowerCase());
            if (StringUtils.isBlank(dataIoName)){
                throw new AtlasException("不能找到数据类型!");
            }

            String nDataName = AtlasDataName.TEMP_LAYER_PREFIX() + "_in_i_" + UUIDUtils.getUUID();
            String url = layerSource.substring(maoHaoIndex + 1);
            Map<String, String> loadParams = generalLoadParams(format, url);
            loadParams.putAll(extraParams);
            String message = client.getSparkRpcClientApi()
                    .loadData(dataIoName, AtlasTagUtils.getAtlasRpcDataTag(nDataName),
                            format, loadParams, taskId, jobId);
            log.info(message);
            return nDataName;
        }

        return layerSource;
    }

    /**
     * 通过存储信息生成载入数据参数
     *
     * @param dataFormat 数据存储格式
     * @param url 数据存储连接
     * @return 载入数据参数
     */
    public Map<String, String> generalLoadParams(String dataFormat, String url) {
        Map<String, String> loadParams = new HashMap<>();
        switch (dataFormat) {
//            case LayerMetadata.SHAPEFILE:
//                /**
//                 * 执行到LayerMetadata.GW_VECTOR
//                 */
//            case LayerMetadata.FILEGDB:
//                /**
//                 * 执行到LayerMetadata.GW_VECTOR
//                 */
//            case LayerMetadata.JDBC:
//                /**
//                 * 执行到LayerMetadata.GW_VECTOR
//                 * 默认如果是LayerMetadata.SHAPEFILE/LayerMetadata.FILEGDB/LayerMetadata.JDBC都是按照默认参数进行数据分区
//                 */
////                loadParams.put(AtlasDataParams.ATLAS_PARTITION_MODE(), "number_per_partition");
////                loadParams.put(AtlasDataParams.ATLAS_PARTITION_PARAMS(), jdbcPartitionNum.toString());
//            case LayerMetadata.GEOJSON:/**
//             * 执行到LayerMetadata.GEOJSON
//             */
//            case LayerMetadata.GRAPH:
//            case LayerMetadata.GDAL_RASTER:
//            case LayerMetadata.GEOTIFF:
//
//            case LayerMetadata.GW_VECTOR:
//            case LayerMetadata.PARQUET:
//                /**
//                 * 执行到LayerMetadata.GW_VECTOR
//                 *
//                 * 如果是GEOJSON或者GW_VECTOR按照默认的分区方式进行分区，仅输入数据路径即可
//                 */
//                loadParams.put(AtlasDataParams.ATLAS_DATA_URL(), url);
//                break;
            case LayerMetadata.SPARK:
                /**
                 * 直接返回
                 */
                break;
            default:
                loadParams.put(AtlasDataParams.ATLAS_DATA_URL(), url);
//                throw new NotSupportException("不支持当前的数据格式:" + dataFormat, Thread.currentThread(), 3);
        }

        return loadParams;
    }

    public String generalResultLayerName(String layerUrl) {
        int maoHaoIndex = layerUrl.indexOf(":");
        if (maoHaoIndex == -1) {
            if (StringUtils.isEmpty(layerUrl)) {
                throw new RuntimeException("结果图层名称不能为空!");
            }
            return layerUrl;
        } else {
            String format = layerUrl.substring(0, maoHaoIndex);
            /**
             * 检查参数是否正确
             */
            if (StringUtils.equalsIgnoreCase(format, LayerMetadata.SPARK)) {
                return layerUrl.substring(maoHaoIndex + 1);
            } else {
                return layerUrl;
            }
        }
    }

    @Override
    public String generalMemoryLayerName(String layerUrl) {
        return generalResultLayerName(layerUrl);
    }

    private void saveLayer(String identity, String persistUrls, Map<String, String> extraSaveParams, String taskId, String jobId) {
        if (StringUtils.isNotEmpty(persistUrls)) {
            String[] persistUrlArray = StringUtils.split(persistUrls, ";");
            for (String persistUrl : persistUrlArray) {
                Map<String, String> saveParams = new HashMap<>();
                int maoHaoIndex = persistUrl.indexOf(":");
                if (maoHaoIndex == -1) {
                    throw new NotFoundException("无法解析成果库地址: " + persistUrl, Thread.currentThread(), 3);
                }
                String format = persistUrl.substring(0, maoHaoIndex);

                if (StringUtils.startsWithIgnoreCase(format, LayerMetadata.SECRET)) {
                    int secretType = Integer.parseInt(StringUtils.substring(format, LayerMetadata.SECRET.length()));
                    String rawPath = SecurityUtils.dencrypt(StringUtils.substring(persistUrl, maoHaoIndex + 1), secretType);
                    saveLayer(identity, rawPath, extraSaveParams, taskId, jobId);
                    return;
                }

                String dataIoName = constants.getDataIoName(format.toLowerCase());
                if (StringUtils.isBlank(dataIoName)) {
                    throw new AtlasException("不能找到数据类型!");
                }

                saveParams.put(AtlasDataParams$.MODULE$.ATLAS_DATA_URL(), persistUrl.substring(maoHaoIndex + 1));
                if (extraSaveParams != null && !extraSaveParams.isEmpty()) {
                    saveParams.putAll(extraSaveParams);
                }
                String message =
                        client.getSparkRpcClientApi()
                                .saveData(dataIoName,
                                        AtlasTagUtils.getAtlasRpcDataTag(identity),
                                        format,
                                        saveParams,
                                        taskId,
                                        jobId);
                log.info(message);
            }
        } else {
            throw new NotFoundException("无法解析成果库地址: " + persistUrls, Thread.currentThread(), 3);
        }
    }

    @Override
    public void saveVectorLayer(String identity, String persistUrls, Map<String, String> extraSaveParams, String taskId, String jobId) {
        saveLayer(identity, persistUrls, extraSaveParams, taskId, jobId);
    }

    @Override
    public void saveGraphLayer(String identity, String persistUrls, Map<String, String> extraSaveParams, String taskId, String jobId) {
        saveLayer(identity, persistUrls, extraSaveParams, taskId, jobId);
    }

    @Override
    public void saveRasterLayer(String identity, String persistUrls, Map<String, String> extraSaveParams, String taskId, String jobId) {
        saveLayer(identity, persistUrls, extraSaveParams, taskId, jobId);
    }

    @Override
    public void saveVectorLayer(String identity, String persistUrl, String taskId, String jobId) {
        saveVectorLayer(identity, persistUrl, new HashMap<>(), taskId, jobId);
    }

    @SneakyThrows
    public String toJSON(Object obj) {
        return objectMapper.writeValueAsString(obj);
    }

    @SneakyThrows
    public <T> T fromJSON(String jsonString, Class<T> clz) {
        return objectMapper.readValue(jsonString, clz);
    }
}
