package com.geoway.atlas.web.api.v2.controller;

import com.geoway.atlas.web.api.v2.exception.ExceptionDeal;
import com.geoway.atlas.web.api.v2.job.AsyncDeal;
import com.geoway.atlas.web.api.v2.service.DataServer;
import com.geoway.atlas.web.api.v2.utils.MapUtils;
import com.geoway.atlas.web.api.v2.utils.RequestUtils;
import com.geoway.atlas.web.api.v2.utils.TaskUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.Map;

/**
 * 数据接口
 *
 * @author zhaotong 2022/9/7 16:53
 */

@Api(value = "Atlas数据相关接口", tags = "数据接口")
@RequestMapping("/api/v2/data")
@RestController
@CrossOrigin(
        origins = {"*"}
)
public class DataController {

    @Autowired
    private DataServer dataServer;

    @ApiOperation("载入矢量数据接口")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "dataStoreFormat", value = "数据格式", required = true),
            @ApiImplicitParam(name = "atlas_data_url", value = "载入数据的路径", required = true),
            @ApiImplicitParam(name = "atlas_data_load_name", value = "数据名称"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @GetMapping(value = {"/vector/load/{dataStoreFormat}"}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> loadVector(
            @PathVariable String dataStoreFormat,
            @RequestParam(value = "atlas_data_url") String dataUrl,
            @RequestParam(value = "atlas_data_load_name") String dataName,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps, Collections.singletonList("atlas.data.load.name"));
        return dataServer.loadVector(dataName, dataStoreFormat, paramMaps, nTaskId, jobId);
    }


    @ApiOperation("输出矢量数据接口")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "dataStoreFormat", value = "数据格式", required = true),
            @ApiImplicitParam(name = "atlas_data_url", value = "输出数据的路径", required = true),
            @ApiImplicitParam(name = "atlas_data_load_name", value = "数据名称"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @GetMapping(value = {"/vector/save/{dataStoreFormat}"}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> saveVector(
            @PathVariable String dataStoreFormat,
            @RequestParam(value = "atlas_data_url") String dataUrl,
            @RequestParam(value = "atlas_data_load_name") String dataName,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps, Collections.singletonList("atlas.data.load.name"));
        return dataServer.saveVector(dataName, dataStoreFormat, paramMaps, nTaskId, jobId);
    }


    @ApiOperation("查看数据")
    @GetMapping(value = {"/_show"}, produces = MediaType.APPLICATION_JSON_VALUE)
    @ExceptionDeal
    public Map<String, Object> showVector() {
        return dataServer.showVector();
    }

    @ApiOperation("数据详情")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas.data.desc.name", value = "数据名称"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @GetMapping(value = {"/_desc"}, produces = MediaType.APPLICATION_JSON_VALUE)
    @ExceptionDeal
    public Map<String, Object> descVector(
            @RequestParam(value = "atlas.data.desc.name") String dataName,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId) {
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        return dataServer.descVector(dataName, nTaskId, jobId);
    }

    @ApiOperation("数据是否存在")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas.data.desc.name", value = "数据名称"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @GetMapping(value = {"/_exist"}, produces = MediaType.APPLICATION_JSON_VALUE)
    @ExceptionDeal
    public Map<String, Object> existVector(
            @RequestParam(value = "atlas.data.desc.name") String dataName,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId) {
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        return dataServer.existVector(dataName, nTaskId, jobId);
    }
}
