package com.geoway.atlas.web.api.v2.service.pkg;

import java.util.Map;

/**
 * @author zhaotong 2022/11/1 18:42
 */
public interface SpatialAnalysisServer {

    /**
     * 求交算子
     *
     * @param baseIdentity        基础图层标识
     * @param baseFilter          基础图层筛选条件
     * @param recomputeBaseFields 重算基础字段名称
     * @param otherIdentity       叠加图层标识
     * @param otherFilter         叠加筛选条件
     * @param intersectionParams  叠加参数
     * @param resultIdentity      结果图层标识
     * @param persistDisk         是否持久化到磁盘
     * @param taskId              任务Id
     * @param jobId               jobId
     * @return 返回union的结果
     */
    Map<String, Object> intersectionProcess(String baseIdentity,
                                            String baseFilter,
                                            String recomputeBaseFields,
                                            String otherIdentity,
                                            String otherFilter,
                                            Map<String, String> intersectionParams,
                                            String resultIdentity,
                                            boolean persistDisk,
                                            String resultUrl,
                                            String taskId,
                                            String jobId);

    /**
     * 擦除算子
     *
     * @param baseIdentity        基础图层标识
     * @param baseFilter          基础图层筛选条件
     * @param recomputeBaseFields 重算基础字段名称
     * @param otherIdentity       叠加图层标识
     * @param otherFilter         叠加筛选条件
     * @param intersectionParams  叠加参数
     * @param resultIdentity      结果图层标识
     * @param persistDisk         是否持久化到磁盘
     * @param taskId              任务Id
     * @param jobId               jobId
     * @return 返回union的结果
     */
    Map<String, Object> eraseProcess(String baseIdentity,
                                     String baseFilter,
                                     String recomputeBaseFields,
                                     String otherIdentity,
                                     String otherFilter,
                                     Map<String, String> intersectionParams,
                                     String resultIdentity,
                                     boolean persistDisk,
                                     String resultUrl,
                                     String taskId,
                                     String jobId);

    /**
     * 更新算子
     *
     * @param baseIdentity        基础图层标识
     * @param baseFilter          基础图层筛选条件
     * @param recomputeBaseFields 重算基础字段名称
     * @param otherIdentity       叠加图层标识
     * @param otherFilter         叠加筛选条件
     * @param intersectionParams  叠加参数
     * @param resultIdentity      结果图层标识
     * @param persistDisk         是否持久化到磁盘
     * @param taskId              任务Id
     * @param jobId               jobId
     * @return 返回union的结果
     */
    Map<String, Object> updateProcess(String baseIdentity,
                                     String baseFilter,
                                     String recomputeBaseFields,
                                     String otherIdentity,
                                     String otherFilter,
                                     Map<String, String> intersectionParams,
                                     String resultIdentity,
                                     boolean persistDisk,
                                     String resultUrl,
                                     String taskId,
                                     String jobId);

    /**
     * 裁切算子
     *
     * @param baseIdentity        基础图层标识
     * @param baseFilter          基础图层筛选条件
     * @param recomputeBaseFields 重算基础字段名称
     * @param otherIdentity       裁切图层标识
     * @param otherFilter         裁切筛选条件
     * @param intersectionParams  裁切参数
     * @param resultIdentity      结果图层标识
     * @param persistDisk         是否持久化到磁盘
     * @param taskId              任务Id
     * @param jobId               jobId
     * @return 返回union的结果
     */
    Map<String, Object> clipProcess(String baseIdentity,
                                    String baseFilter,
                                    String recomputeBaseFields,
                                    String otherIdentity,
                                    String otherFilter,
                                    Map<String, String> intersectionParams,
                                    String resultIdentity,
                                    boolean persistDisk,
                                    String resultUrl,
                                    String taskId,
                                    String jobId);

    /**
     * 联合算子
     *
     * @param baseIdentity         基础图层标识
     * @param baseFilter           基础图层筛选条件
     * @param recomputeBaseFields  重算基础字段名称
     * @param unionIdentity        联合图层标识
     * @param unionFilter          联合筛选条件
     * @param recomputeUnionFields 重算联合字段名称
     * @param unionParams          联合参数
     * @param resultIdentity       结果图层标识
     * @param persistDisk          是否持久化到磁盘
     * @param taskId               任务Id
     * @param jobId                jobId
     * @return 返回union的结果
     */
    Map<String, Object> unionProcess(String baseIdentity,
                                     String baseFilter,
                                     String recomputeBaseFields,
                                     String unionIdentity,
                                     String unionFilter,
                                     String recomputeUnionFields,
                                     Map<String, String> unionParams,
                                     String resultIdentity,
                                     boolean persistDisk,
                                     String resultUrl,
                                     String taskId,
                                     String jobId);

    /**
     * 标识算子
     *
     * @param baseIdentity        基础图层标识
     * @param baseFilter          基础图层筛选条件
     * @param recomputeBaseFields 重算基础字段名称
     * @param otherIdentity       其他图层标识
     * @param otherFilter         其他图层筛选条件
     * @param identityParams      标识参数
     * @param resultIdentity      结果图层标识
     * @param persistDisk         是否持久化到磁盘
     * @param resultUrl           持久化磁盘的url
     * @param taskId              任务Id
     * @param jobId               jobId
     * @return 返回标识的结果
     */
    Map<String, Object> identityProcess(String baseIdentity,
                                        String baseFilter,
                                        String recomputeBaseFields,
                                        String otherIdentity,
                                        String otherFilter,
                                        String recomputeOtherFields,
                                        Map<String, String> identityParams,
                                        String resultIdentity,
                                        boolean persistDisk,
                                        String resultUrl,
                                        String taskId,
                                        String jobId);

    // dissolveProcess

    /**
     * 裁切算子
     *
     * @param baseIdentity   基础图层标识
     * @param dissolveParams 融合参数
     * @param resultIdentity 结果图层标识
     * @param persistDisk    是否持久化到磁盘
     * @param taskId         任务Id
     * @param jobId          jobId
     * @return 返回union的结果
     */
    Map<String, Object> dissolveProcess(String baseIdentity,
                                        Map<String, String> dissolveParams,
                                        String resultIdentity,
                                        boolean persistDisk,
                                        String resultUrl,
                                        String taskId,
                                        String jobId);

    /**
     * 去重计算
     *
     * @param baseIdentity    基础图层标识
     * @param recomputeFields 重算面积字段
     * @param resultIdentity  结果标识
     * @param persistDisk     是否持久化到磁盘
     * @param resultUrl       持久化路径
     * @param taskId          任务Id
     * @param jobId           jobId
     * @return 返回响应结果
     */
    Map<String, Object> distinctProcess(String baseIdentity,
                                        String recomputeFields,
                                        String resultIdentity,
                                        boolean persistDisk,
                                        String resultUrl,
                                        String taskId,
                                        String jobId);

    /**
     * 连通性分析
     *
     * @param baseIdentity       基础图层标识
     * @param connectivityParams 连通参数
     * @param resultIdentity     结果图层标识
     * @param persistDisk        是否持久化到磁盘
     * @param taskId             任务Id
     * @param jobId              jobId
     * @return 返回连通的结果
     */
    Map<String, Object> connectivityProcess(String baseIdentity,
                                            Map<String, String> connectivityParams,
                                            String resultIdentity,
                                            boolean persistDisk,
                                            String resultUrl,
                                            String taskId,
                                            String jobId);

    /**
     * 缓冲区分析
     *
     * @param baseIdentity   基础图层名称
     * @param bufferParams   缓冲参数
     * @param dissolve_type  融合类型
     * @param resultIdentity 结果图层名称
     * @param persistDisk    是否持久化磁盘
     * @param resultUrl      结果路径
     * @param taskId         任务id
     * @param jobId          jobid
     * @return
     */
    Map<String, Object> bufferProcess(String baseIdentity,
                                      Map<String, String> bufferParams,
                                      String dissolve_type,
                                      String resultIdentity,
                                      boolean persistDisk,
                                      String resultUrl,
                                      String taskId,
                                      String jobId);

    /**
     * 网格剖分
     *
     * @param baseIdentity   基础图层标识
     * @param splitParams    网格剖分参数
     * @param resultIdentity 结果图层标识
     * @param persistDisk    是否持久化成果库
     * @param resultUrl      成果库地址
     * @param taskId         任务iD
     * @param jobId          工作项Id
     * @return 返回响应结果
     */
    Map<String, Object> gridSplit(String baseIdentity,
                                  Map<String, String> splitParams,
                                  String resultIdentity,
                                  boolean persistDisk,
                                  String resultUrl,
                                  String taskId,
                                  String jobId);

    /**
     * 叠加赋值
     *
     * @param baseIdentity   基础图层标识
     * @param baseFilter     基础图层筛选条件
     * @param otherIdentity  叠加图层标识
     * @param otherFilter    叠加图层的筛选条件
     * @param assignFields   赋值字段名称
     * @param assignType     赋值类型
     * @param isConsiderNull 空值是否参与赋值
     * @param defaultValue   默认值
     * @param resultIdentity 结果标识
     * @param persistDisk    是否持久化磁盘
     * @param resultUrl      结果url
     * @param taskId         任务Id
     * @param jobId          jobId
     * @return 返回结果
     */
    Map<String, Object> assignProcess(
            String baseIdentity, String baseFilter,
            String otherIdentity, String otherFilter,
            String assignFields,
            String assignType,
            boolean isConsiderNull,
            String defaultValue,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId, String jobId);
}
