package com.geoway.atlas.web.api.v2.service.pkg.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.geoway.atlas.common.error.*;
import com.geoway.atlas.common.utils.UUIDUtils;
import com.geoway.atlas.data.common.data.AtlasDataName;
import com.geoway.atlas.data.vector.spark.common.rpc.AtlasRpcDataTag;
import com.geoway.atlas.data.vector.spark.common.rpc.RpcDescDataRespond;
import com.geoway.atlas.function.parser.common.FunctionDSLParser;
import com.geoway.atlas.web.api.v2.component.rpc.RpcClientProxy;
import com.geoway.atlas.data.vector.spark.common.rpc.common.Constants;
import com.geoway.atlas.web.api.v2.dto.LayerIdentityDto;
import com.geoway.atlas.web.api.v2.dto.pack.ComputeLayerInfo;
import com.geoway.atlas.web.api.v2.dto.pack.CreateOidLayerInfo;
import com.geoway.atlas.web.api.v2.dto.vector.FieldInfo;
import com.geoway.atlas.web.api.v2.job.JobManager;
import com.geoway.atlas.web.api.v2.service.DataServer;
import com.geoway.atlas.web.api.v2.service.pkg.SpatialAnalysisServer;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.geoway.atlas.web.api.v2.service.ProcessServer;
import com.geoway.atlas.web.api.v2.service.pkg.impl.assigin.AssignFunctionPlan;
import com.geoway.atlas.web.api.v2.utils.*;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaotong 2022/11/1 18:46
 */
@Slf4j
@Service
public class SpatialAnalysisServerImpl implements SpatialAnalysisServer {

    private static final String BAK_FIELD_SUFFIX = "_bak";

    private static final String AREA_FIELD_PREFIX = "area";
    private static final String OID_FIELD_PREFIX = "oid";

    private static final String DISSOLVE_TYPE_NONE = "NONE";
    private static final String DISSOLVE_TYPE_GRID = "GRID";
    private static final String DISSOLVE_TYPE_CONNECTIVITY = "CONNECT";

    private static final Double OVERLAY_OPTIMIZE_RATE = 0.5;

    @Autowired
    private Constants constants;

    @Autowired
    private DataServer dataServer;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private RpcClientProxy client;

    @Autowired
    private ProcessServer processServer;

    @Autowired
    private JobManager jobManager;

    @Override
    public Map<String, Object> intersectionProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String otherIdentity, String otherFilter, Map<String, String> intersectionParams,
            String resultIdentity, boolean persistDisk, String resultUrl, String taskId, String jobId) {
        String processName = constants.intersectionProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                null,
                processName,
                intersectionParams,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合求交流程!");
    }


    @Override
    public Map<String, Object> clipProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String otherIdentity, String otherFilter, Map<String, String> intersectionParams,
            String resultIdentity, boolean persistDisk, String resultUrl, String taskId, String jobId) {
        String processName = constants.clipProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                null,
                processName,
                intersectionParams,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合裁切流程!");
    }

    @Override
    public Map<String, Object> eraseProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String otherIdentity, String otherFilter, Map<String, String> intersectionParams,
            String resultIdentity, boolean persistDisk, String resultUrl, String taskId, String jobId) {
        String processName = constants.eraseProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                null,
                processName,
                intersectionParams,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合擦除流程!");
    }

    @Override
    public Map<String, Object> updateProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String otherIdentity, String otherFilter, Map<String, String> intersectionParams,
            String resultIdentity, boolean persistDisk, String resultUrl, String taskId, String jobId) {
        String processName = constants.updateProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                null,
                processName,
                intersectionParams,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合擦除流程!");
    }

    @Override
    public Map<String, Object> unionProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String unionIdentity, String otherFilter, String recomputeOtherFields,
            Map<String, String> params,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId, String jobId) {

        String processName = constants.unionProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(unionIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(unionIdentity, otherLayerName),
                otherFilter,
                recomputeOtherFields,
                processName,
                params,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合联合流程!");
    }

    @Override
    public Map<String, Object> assignProcess(
            String baseIdentity, String baseFilter,
            String otherIdentity, String otherFilter,
            String assignFields,
            String assignType,
            boolean isConsiderNull,
            String defaultValue,
            String resultIdentity,
            boolean persistDisk,
            String resultUrl,
            String taskId, String jobId) {

        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlayAssign(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                assignFields,
                assignType,
                isConsiderNull,
                defaultValue,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合叠加赋值流程!");
    }

    @Override
    public Map<String, Object> identityProcess(
            String baseIdentity, String baseFilter, String recomputeBaseFields,
            String otherIdentity, String otherFilter, String recomputeOtherFields,
            Map<String, String> identityParams, String resultIdentity, boolean persistDisk, String resultUrl,
            String taskId, String jobId) {

        String processName = constants.identityProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getSourceFilterParams(baseFilter), taskId, UUIDUtils.getUUID());
        String otherLayerName = dataServer.loadVectorLayer(otherIdentity, InputParamUtils.getSourceFilterParams(otherFilter), taskId, UUIDUtils.getUUID());
//        LayerIdentityDto = fromJSON(resultIdentity, LayerIdentityDto.class);
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        overlay(baseLayerName,
                StringUtils.equals(baseIdentity, baseLayerName),
                baseFilter,
                recomputeBaseFields,
                otherLayerName,
                StringUtils.equals(otherIdentity, otherLayerName),
                otherFilter,
                recomputeOtherFields,
                processName,
                identityParams,
                resultLayerName,
                taskId);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合标识分析流程!");
    }

    @Override
    public Map<String, Object> dissolveProcess(String baseIdentity,
                                               Map<String, String> dissolveParams,
                                               String resultIdentity,
                                               boolean persistDisk,
                                               String resultUrl,
                                               String taskId,
                                               String jobId) {
        String processName = constants.dissolveProcessName();
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(dissolveParams), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName, dissolveParams, AtlasTagUtils.getAtlasRpcDataTag(resultLayerName), taskId, jobId);

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合融合分析流程!");
    }

    public Map<String, Object> distinctProcess(String baseIdentity,
                                               String recomputeFields,
                                               String resultIdentity,
                                               boolean persistDisk,
                                               String resultUrl,
                                               String taskId,
                                               String jobId) {
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(new HashMap<>()), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        distinct(baseLayerName, recomputeFields, taskId, resultLayerName);
        log.info("完成去重计算");
        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合去重流程!");
    }

    public Map<String, Object> connectivityProcess(String baseIdentity,
                                                   Map<String, String> connectivityParams,
                                                   String resultIdentity,
                                                   boolean persistDisk,
                                                   String resultUrl,
                                                   String taskId,
                                                   String jobId) {
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(connectivityParams), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        PluginParallelManager.runPlugin("connectivity", new Function<Object[], Void>() {
            @Override
            public Void apply(Object[] objects) {
                connectivity((String) objects[0], (Map<String, String>) objects[1], (String) objects[2], (String) objects[3]);
                return null;
            }
        }, new Object[]{baseLayerName, connectivityParams, resultLayerName, taskId}, jobManager, jobId, taskId);


        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合连通性分析流程!");
    }

    @Override
    public Map<String, Object> bufferProcess(String baseIdentity,
                                             Map<String, String> bufferParams,
                                             String dissolve_type,
                                             String resultIdentity,
                                             boolean persistDisk,
                                             String resultUrl,
                                             String taskId,
                                             String jobId) {
        // 获取原始数据的图层名称相关信息
        String baseLayerName = dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(bufferParams), taskId, UUIDUtils.getUUID());
        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        if (DISSOLVE_TYPE_CONNECTIVITY.equalsIgnoreCase(dissolve_type)) {
            PluginParallelManager.runPlugin("connectivity", (Function<Object[], Void>) objects -> {
                buffer((String) objects[0], (Map<String, String>) objects[1], (String) objects[2], (String) objects[3], (String) objects[4]);
                return null;
            }, new Object[]{baseLayerName, bufferParams, dissolve_type, resultLayerName, taskId}, jobManager, jobId, taskId);
        } else {
            buffer(baseLayerName, bufferParams, dissolve_type, resultLayerName, taskId);
        }

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成组合缓冲分析流程!");
    }

    @Override
    public Map<String, Object> gridSplit(String baseIdentity,
                                         Map<String, String> splitParams,
                                         String resultIdentity,
                                         boolean persistDisk,
                                         String resultUrl,
                                         String taskId,
                                         String jobId) {


        String processName = constants.gridSplitProcessName();

        String baseLayerName =
                dataServer.loadVectorLayer(baseIdentity, InputParamUtils.getInputVectorParams(splitParams), taskId, UUIDUtils.getUUID());

        String resultLayerName = dataServer.generalResultLayerName(resultIdentity);

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseLayerName),
                processName,
                splitParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId, UUIDUtils.getUUID());

        log.info(message);

        if (persistDisk) {
            dataServer.saveVectorLayer(resultLayerName, resultUrl, taskId, jobId);
        }

        return ResponseBuilder.buildSuccess("完成网格剖分!");
    }

    public void overlayAssign(String baseLayerName,
                              boolean baseIsMem,
                              String baseFilter,
                              String otherLayerName,
                              boolean otherIsMem,
                              String otherFilter,
                              String assignFields,
                              String assignType,
                              boolean isConsiderNull,
                              String defaultValues,
                              String resultLayerName,
                              String taskId) {

        log.info("进入叠加赋值流程, 参数为 => baseFilter:" + baseFilter +
                ", otherFilter:" + otherFilter +
                ", assignFields:" + assignFields +
                ", assignType:" + assignType +
                ", isConsiderNull:" + toJSON(isConsiderNull) +
                ", defaultValues:" + defaultValues);

        if (StringUtils.isBlank(assignFields)) {
            throw new ParamException("必须设置叠加赋值字段名称!", Thread.currentThread(), 3);
        }
        // 保证key插入顺序
        Map<String, String> fieldMap = new LinkedHashMap<>();
        Map<String, String> defaultValueMap = new LinkedHashMap<>();
        parseAssignParams(assignFields, defaultValues, fieldMap, defaultValueMap);
        log.info("赋值字段映射:" + toJSON(fieldMap));
        log.info("赋值字段默认值映射:" + toJSON(defaultValueMap));

        // 获取原始图层的信息
        RpcDescDataRespond baseLayerInfo = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(baseLayerName), taskId, UUIDUtils.getUUID());
        Map<String, String> baseLayerDescMap = baseLayerInfo.getDataDescMap();
        List<String> baseLayerFieldNames = RpcDescRespondUtils.getVectorFieldInfo(baseLayerInfo)
                .stream()
                .map(FieldInfo::getFieldName)
                .collect(Collectors.toList());
        if (CollectionUtils.containsAny(baseLayerFieldNames, fieldMap.values())) {
            throw new DuplicateException("原始图层字段包含赋值后的字段，请重新设置赋值字段映射", Thread.currentThread(), 3);
        }

        if (StringUtils.isBlank(baseLayerDescMap.get(constants.dataDescDefaultGeometry()))) {
            throw new NotSupportException("不支持非空间图层进行叠加赋值操作", Thread.currentThread(), 3);
        }

        // 获取赋值图层信息
        RpcDescDataRespond otherLayerInfo = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(otherLayerName), taskId, UUIDUtils.getUUID());

        String bFilterLayerName = filter(baseLayerName, baseFilter, taskId, UUIDUtils.getUUID());
        String oFilterLayerName;

        if (!otherIsMem) {
            String extraFilter = GeometryBoundaryUtils.getExtraFilter(baseLayerInfo, otherLayerInfo, OVERLAY_OPTIMIZE_RATE)
                    .getOrDefault(1, null);
            log.info("赋值图层新增空间筛选条件: " + extraFilter);
            // 进行属性筛选
            oFilterLayerName = filter(otherLayerName, GeometryBoundaryUtils.addExtraFilter(otherFilter, extraFilter), taskId, UUIDUtils.getUUID());
        } else {
            oFilterLayerName = filter(otherLayerName, otherFilter, taskId, UUIDUtils.getUUID());
        }

        log.info("属性筛选后的基础图层为:" + bFilterLayerName);
        log.info("属性筛选后的叠加图层为:" + oFilterLayerName);

        // 获取坐标转换后的图层
        String otherTransLayerName = transFormCrs(oFilterLayerName,
                baseLayerDescMap.get(constants.dataDescCrs()), taskId, UUIDUtils.getUUID());

        log.info("坐标转换后的叠加图层为:" + otherTransLayerName);

        // 创建oid字段
        CreateOidLayerInfo createOidLayerInfo = createOid(bFilterLayerName, taskId, UUIDUtils.getUUID());

        log.info("针对" + baseLayerName + "进行赋值!");

        String nAssignType = StringUtils.startsWith(assignType.trim(),"{") ?
                (String) fromJSON(assignType, Map.class).get("value") : assignType;

        assignStage(createOidLayerInfo.getLayerName(),
                baseLayerInfo.getDataDescMap(),
                createOidLayerInfo.isDelete(),
                otherTransLayerName,
                otherLayerInfo.getDataDescMap(),
                createOidLayerInfo.getOidName(),
                nAssignType,
                fieldMap,
                isConsiderNull,
                defaultValueMap,
                resultLayerName,
                taskId);

    }

    public void parseAssignParams(String assignFields,
                                  String defaultValues,
                                  Map<String, String> fieldMap,
                                  Map<String, String> defaultValueMap) {
        String[] rawAssignField = StringUtils.split(assignFields, ",");
        String[] defaultValueArray = null;
        if (StringUtils.isNotBlank(defaultValues)) {
            defaultValueArray = StringUtils.split(defaultValues, ",");
        }
        for (int i = 0; i < rawAssignField.length; i++) {
            String key;
            String value;
            if (rawAssignField[i].contains(":")) {
                String[] kv = rawAssignField[i].split(":");
                key = kv[0];
                value = kv[1];
            } else {
                key = rawAssignField[i];
                value = rawAssignField[i];
            }
            fieldMap.put(key, value);
            if (defaultValueArray != null) {
                if(i < defaultValueArray.length) {
                    if(StringUtils.isNotBlank(defaultValueArray[i].trim())){
                        defaultValueMap.put(value, defaultValueArray[i].trim());
                    }else {
                        defaultValueMap.put(value, "");
                    }
                }
            }
        }
    }

    /**
     * 赋值过程
     *
     * max
     * max() 面积最大计算
     *
     * concat_value
     * concat_value()
     * concat_value(order) 排序方式支持0或1 concat_value(int)
     * concat_value(order,precision, 作为结果图层的重算面积字段) 排序方式支持 0或1 重算面积字段支持用户自己指定（指定图层的按比例重算的面积字段） concat_value(int， col)
     *                  concat_value(0, 2, #1.tbdlmj)
     *
     * agg_value(聚合表达式) 聚合表达式 agg_value(string)
     * agg_value(聚合表达式1, 聚合表达式2, 聚合表达式3, 重算字段1，重算字段2。。。。) agg_value(string, col1, col2, .....)
     * agg_value('concat_ws('、',sort_array(collect_set(#1.dlbm)))||':'||round(sum(#1.tbmj),2)', #1.tbmj)
     *
     * sum_area(作为结果图层的重算面积字段) sum_area(#2.`dasf`) 作为结果的重算面积字段名称
     * sum_rate
     * sum_rate()计算压占比例
     *
     * condition_rate('ca')
     * condition_rate('#rate', 'ca')
     * condition_rate(阈值, 小于赋值 大于赋值) 按照叠加总面积比例判断 condition(double, string, string)
     *
     * @param layerName       基础图层名称
     * @param deleteOid       是否删除oid字段
     * @param otherTransLayerName  赋值图层
     * @param rawOidName          oid字段名称
     * @param assignType      赋值类型
     * @param assignFieldMap  赋值的字段及映射
     * @param isConsiderNull  是否考虑空值
     * @param defaultValueMap 默认值映射
     * @param resultLayer     结果图层名称
     * @param taskId          任务id
     */
    public void assignStage(String layerName,
                            Map<String, String> rawBaseLayerParams,
                            boolean deleteOid,
                            String otherTransLayerName,
                            Map<String, String> rawOtherLayerParams,
                            String rawOidName,
                            String assignType,
                            Map<String, String> assignFieldMap,
                            boolean isConsiderNull,
                            Map<String, String> defaultValueMap,
                            String resultLayer,
                            String taskId) {

        // 获取赋值类型
        AssignFunctionPlan functionPlan = (AssignFunctionPlan) FunctionDSLParser.parser(assignType);
        functionPlan.setDefaultValueMap(defaultValueMap);
        functionPlan.setConsiderNull(isConsiderNull);

        String persistLayerName = layerName;
        // 如果需要计算图斑的shapearea，那么原始图形的面积字段也需要计算，以便计算比例相关信息
        if(functionPlan.isNeedShapeArea()){
            // 添加面积字段
            ComputeLayerInfo computeLayerInfo = computeArea(layerName, true, taskId, UUIDUtils.getUUID());
            persistLayerName = computeLayerInfo.getComputeLayerName();
            functionPlan.setBaseShapeAreaName(computeLayerInfo.getAreaName());
        }
        // 持久化基础图层
        client.getSparkRpcClientApi()
                .persistData(AtlasTagUtils.getAtlasRpcDataTag(persistLayerName), taskId, UUIDUtils.getUUID());
        log.info("完成" + persistLayerName + "数据持久化!");
        // 获取执行结果
        String processName = constants.intersectionProcessName();
        String preMiddleLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_r_" + UUIDUtils.getUUID();
        Map<String, String> iProcessParams = new HashMap<>();
        if(!functionPlan.needRepair()) {
            iProcessParams.put(constants.intersectionIsRepair(), "false");
        }
        // **为赋值图层添加固定后缀**
        iProcessParams.put(constants.intersectionRightSuffix(), AssignFunctionPlan.FIXED_ASSIGN_SUFFIX);

        // 基础图层重算面积字段名称和赋值图层重算面积字段名称
        List<String> baseRecomputeFields = new ArrayList<>();
        List<String> assignRecomputeFields = new ArrayList<>();

        if(functionPlan.isNeedRecomputeArea()){
            baseRecomputeFields = functionPlan.getLeftRecomputeFields();
            assignRecomputeFields = functionPlan.getRightRecomputeFields();
        }

        overlayExecute(persistLayerName, rawBaseLayerParams,
                StringUtils.join(baseRecomputeFields, ","),
                otherTransLayerName,
                StringUtils.join(assignRecomputeFields, ","),
                processName, iProcessParams, preMiddleLayer, taskId);

        log.info("叠加执行结果图层为:" + preMiddleLayer);

        RpcDescDataRespond descDataRespond = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(preMiddleLayer), taskId, UUIDUtils.getUUID());

        List<FieldInfo> fieldInfos = RpcDescRespondUtils.getVectorFieldInfo(descDataRespond);

        List<String> aFields = fieldInfos.stream().map(FieldInfo::getFieldName).collect(Collectors.toList());
        log.info("统计表的全部字段为:" + StringUtils.join(aFields, ","));

        // 获取图斑字段名称
        String geometryField = descDataRespond.getDataDescMap().get(constants.dataDescDefaultGeometry());
        if (StringUtils.isBlank(geometryField)) {
            throw new NotFoundException("不能找到图形字段!", Thread.currentThread(), 3);
        }

        // 仅保留函数相关字段
        List<String> removeFields = functionPlan.unusedFields(aFields, assignFieldMap);
        removeFields.remove(rawOidName);

        String computeOptLayerName = preMiddleLayer;

        // 如果没有重算面积则计算图形面积并为funtionplan赋值
        if(functionPlan.isNeedShapeArea()){
            // 添加面积字段
            ComputeLayerInfo computeLayerInfo = computeArea(preMiddleLayer, true, taskId, UUIDUtils.getUUID());
            computeOptLayerName = computeLayerInfo.getComputeLayerName();
            functionPlan.setIntersectShapeAreaName(computeLayerInfo.getAreaName());
        }

        String middleLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_mid_" + UUIDUtils.getUUID();
        // 删除字段
        Map<String, String> processParams = new HashMap<>();
        processParams.put(constants.fieldRemoveProcessRmFields(), StringUtils.join(removeFields, ","));
        client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(computeOptLayerName),
                constants.fieldRemoveProcessName(), processParams,
                AtlasTagUtils.getAtlasRpcDataTag(middleLayer), taskId, UUIDUtils.getUUID());

        // 持久化基础图层
        client.getSparkRpcClientApi()
                .persistData(AtlasTagUtils.getAtlasRpcDataTag(middleLayer), taskId, UUIDUtils.getUUID());
        log.info("完成" + middleLayer + "数据持久化，" + "字段包含：" +
                RpcDescRespondUtils.getVectorFieldInfo(client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(middleLayer), taskId, UUIDUtils.getUUID())).stream().map(FieldInfo::getFieldName)
                .collect(Collectors.joining(",")));

        /**
         * (select oid,`groupNames`, sum(`area`) as `area` from middleLayer groupby oid,`groupNames` as t)
         */
        String middleLayerView = RpcNameUtils.getViewName(middleLayer);

        String statisticSql = functionPlan.getStatisticSql(middleLayerView, assignFieldMap.keySet(), rawOidName);

        /**
         * select
         * a.oid,
         * a.`groupNames`,
         * row_number() over(partition by a.oid order by a.`area` desc) as rowkey （每个oid中分组面积最大的row number为1，后续会把rownumber为1的值挂接过去）,
         * sum(a.area) over(partition by a.oid) sumarea (所有压占面积总和),
         * a.area assignarea,
         * from statistic_sql a
         * select b.oid, b.`groupNames`, b.assignarea, b.sumarea from group_sql b where b.rowkey=1
         */

        String assignSql = functionPlan.getAssignFuncSql(statisticSql, rawOidName, assignFieldMap);

        String filterViewAlias = "c";

        String baseView = RpcNameUtils.getViewName(persistLayerName);
        RpcDescDataRespond baseLayerInfo = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(persistLayerName), taskId, UUIDUtils.getUUID());
        // 获取图斑字段名称
        String baseGeomField = baseLayerInfo.getDataDescMap().get(constants.dataDescDefaultGeometry());
        if (StringUtils.isBlank(baseGeomField)) {
            throw new NotFoundException("不能找到图形字段!", Thread.currentThread(), 3);
        }

        // 获取完整的基础图层字段名称列表
        List<String> baseLayerFields =
                RpcDescRespondUtils.getVectorFieldInfo(baseLayerInfo)
                        .stream()
                        .map(FieldInfo::getFieldName)
                        .collect(Collectors.toList());
        if (deleteOid) {
            baseLayerFields.remove(rawOidName);
        }

        if(functionPlan.isNeedShapeArea()){
            baseLayerFields.remove(functionPlan.getBaseShapeAreaName());
        }

        String baseLayerAlias = "d";

        StringBuilder joinSqlBuilder =
                new StringBuilder("select " + baseLayerAlias + ".`" + StringUtils.join(baseLayerFields, "`," + baseLayerAlias + ".`") + "`,");

        joinSqlBuilder.append(functionPlan.getFieldSelectWithDefExpr(filterViewAlias, assignFieldMap.values()));

        joinSqlBuilder.append(" from " + baseView + " as " + baseLayerAlias +
                " left join (" + assignSql + ") as " + filterViewAlias +
                " on " + baseLayerAlias + ".`" + rawOidName + "` = " +
                filterViewAlias + ".`" + rawOidName + "`");
        String joinSql = joinSqlBuilder.toString();
        log.info("执行SparkSql:" + joinSql);

        Map<String, String> sqlParams = new HashMap<>();
        sqlParams.put(constants.runSqlExpression(), joinSql);
        sqlParams.put(constants.sqlStatisticLayerOrder(), "0");
        String message = client.getSparkRpcClientApi().listProcess(
                Arrays.asList(AtlasTagUtils.getAtlasRpcDataTag(persistLayerName), AtlasTagUtils.getAtlasRpcDataTag(middleLayer)),
                constants.runSqlProcessName(),
                sqlParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayer),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);

    }

//    public String assignCaseWhenIncludeNull(String baseLayerAlias,
//                                            String baseGeomField,
//                                            String assignType,
//                                            String groupViewAlias,
//                                            String groupAreaField,
//                                            String sumAreaField,
//                                            String rawAssignField,
//                                            String finalAssignField,
//                                            String defaultValue) {
//
//        return "case when " + groupViewAlias + ".`" + groupAreaField + "` is null " +
//                "then '" + defaultValue + "' " +
//                "else case when " +
//                "(st_area(" + baseLayerAlias + ".`" + baseGeomField + "`) - " + groupViewAlias + ".`" + sumAreaField + "`) " +
//                AssignType.getSparkSqlCasWhen(assignType) + " " + groupViewAlias + ".`" + groupAreaField + "` " +
//                "then '" + defaultValue + "' " +
//                "else " + groupViewAlias + ".`" + rawAssignField + "` " +
//                "end end as " + finalAssignField;
//    }
//
//    public String assignCaseWhen(String groupViewAlias,
//                                 String groupAreaField,
//                                 String rawAssignField,
//                                 String finalAssignField,
//                                 String defaultValue) {
//
//        return "case when " + groupViewAlias + ".`" + groupAreaField + "` is null " +
//                "then '" + defaultValue + "' " +
//                "else " + groupViewAlias + ".`" + rawAssignField + "` " +
//                "end as " + finalAssignField;
//    }

    public void connectivity(String layerName,
                             Map<String, String> connectivityParams,
                             String resultLayer,
                             String taskId) {

        String message = client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(layerName),
                constants.connectedComponentsProcessName(),
                connectivityParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayer),
                taskId, UUIDUtils.getUUID());

        log.info(message);
    }

    private Map<String, Integer> getBufferDict(){
        return new HashMap<String, Integer>() {{
            // Side Type full,left,right,outside_only
            put("FULL", constants.bufferSideTypeFull());
            put("LEFT", constants.bufferSideTypeLeft());
            put("RIGHT", constants.bufferSideTypeRight());
            put("OUTSIDE_ONLY", constants.bufferSideTypeOutSideOnly());
            // End Type ROUND, FLAT
            put("ROUND", constants.bufferEndTypeRound());
            put("FLAT", constants.bufferEndTypeFlat());
            // DISSOLVE TYPE NONE,GRID,CONNECTIVITY
        }};
    }

    public void buffer(String baseName,
                       Map<String, String> params,
                       String dissolve_type,
                       String resultName,
                       String taskId) {

        Map<String, String> bufferParams = new HashMap<>(params);
        if (bufferParams.containsKey(constants.bufferEndType())) {
            bufferParams.put(constants.bufferEndType(),
                    getBufferDict().get(params.get(constants.bufferEndType())).toString());
        }

        if (bufferParams.containsKey(constants.bufferSideType())) {
            bufferParams.put(constants.bufferSideType(),
                    getBufferDict().get(params.get(constants.bufferSideType())).toString());
        }

        log.info("开始图斑缓冲，" +
                "缓冲参数为:" + toJSON(bufferParams) +
                ", 融合类型" + dissolve_type);

        String bufferLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_b_" + UUIDUtils.getUUID();
        client.getSparkRpcClientApi().unitaryProcess(
                AtlasTagUtils.getAtlasRpcDataTag(baseName),
                constants.bufferProcessName(),
                bufferParams,
                AtlasTagUtils.getAtlasRpcDataTag(bufferLayer),
                taskId,
                UUIDUtils.getUUID());

        if (StringUtils.isNotBlank(dissolve_type) && !StringUtils.equals(DISSOLVE_TYPE_NONE, dissolve_type)) {
            switch (dissolve_type.toUpperCase()) {
                case DISSOLVE_TYPE_CONNECTIVITY:
                    String FINAL_ID = "objectid";
                    String connectivityField = "c_" + UUIDUtils.getUUID();
                    String connectivityLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_c_" + UUIDUtils.getUUID();
                    Map<String, String> connectivityParams = new HashMap<>();
                    connectivityParams.put(constants.connectedComponentsFieldName(), connectivityField);
                    client.getSparkRpcClientApi().unitaryProcess(
                            AtlasTagUtils.getAtlasRpcDataTag(bufferLayer),
                            constants.connectedComponentsProcessName(),
                            connectivityParams,
                            AtlasTagUtils.getAtlasRpcDataTag(connectivityLayer),
                            taskId, UUIDUtils.getUUID());

                    String dissolveLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_d_" + UUIDUtils.getUUID();
                    Map<String, String> dissolveParams = new HashMap<>();
                    dissolveParams.put(constants.dissolveGroupFields(), connectivityField);
                    client.getSparkRpcClientApi().unitaryProcess(
                            AtlasTagUtils.getAtlasRpcDataTag(connectivityLayer),
                            constants.dissolveProcessName(),
                            dissolveParams,
                            AtlasTagUtils.getAtlasRpcDataTag(dissolveLayer),
                            taskId, UUIDUtils.getUUID());
                    Map<String, String> renameFieldParams = new HashMap<>();
                    renameFieldParams.put(constants.fieldRenameOidField(), connectivityField);
                    renameFieldParams.put(constants.fieldRenameNewField(), FINAL_ID);
                    client.getSparkRpcClientApi().unitaryProcess(
                            AtlasTagUtils.getAtlasRpcDataTag(dissolveLayer),
                            constants.fieldRenameProcessName(),
                            renameFieldParams,
                            AtlasTagUtils.getAtlasRpcDataTag(resultName),
                            taskId, UUIDUtils.getUUID());
                    break;
                case DISSOLVE_TYPE_GRID:
                    throw new NotImplementedException("暂未实现当前缓冲方法:" + dissolve_type + "!", Thread.currentThread(), 3);
                default:
                    throw new NotSupportException("不支持当前融合方法:" + dissolve_type + "!", Thread.currentThread(), 3);
            }
        } else {
            dataServer.renameData(bufferLayer, resultName, taskId, UUIDUtils.getUUID());
        }
    }

    public void distinct(String baseLayerName,
                         String recomputeFields,
                         String taskId,
                         String resultLayerName) {

        // 面积计算
        ComputeLayerInfo computerLayerInfo = computeArea(baseLayerName,
                StringUtils.isNotBlank(recomputeFields), taskId, UUIDUtils.getUUID());

        log.info("面积计算后的基础图层为:" + computerLayerInfo.getComputeLayerName());
        log.info("面积计算后基础图层追加的面积字段为:" + computerLayerInfo.getAreaName());

        // 获取执行结果
        String preResultLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_r_" + UUIDUtils.getUUID();
        processServer.unitaryProcess(computerLayerInfo.getComputeLayerName(), "overlay-distinct",
                new HashMap<>(), preResultLayerName, taskId, UUIDUtils.getUUID());
        log.info("叠加执行结果图层为:" + preResultLayerName);
        if (StringUtils.isNotBlank(recomputeFields)) {
            recomputeFields(computerLayerInfo.getComputeLayerName(),
                    computerLayerInfo.getAreaName(),
                    recomputeFields,
                    null,
                    null,
                    null,
                    preResultLayerName,
                    resultLayerName,
                    taskId);
        } else {
            dataServer.renameData(preResultLayerName, resultLayerName, taskId, UUIDUtils.getUUID());
        }
    }

    /**
     * 叠加分析
     *
     * @param baseLayerName        基础图层名称
     * @param baseFilter           基础图层筛选条件
     * @param recomputeBaseFields  重算基础字段名称
     * @param otherLayerName       其他图层名称
     * @param otherFilter          其他图层筛选条件
     * @param recomputeOtherFields 重算其他图层字段名称
     * @param processName          执行器名称
     * @param processParams        执行器参数
     * @param resultLayerName      结果图层名称
     * @param taskId               任务Id
     */
    public void overlay(String baseLayerName,
                        boolean baseIsMem,
                        String baseFilter,
                        String recomputeBaseFields,
                        String otherLayerName,
                        boolean otherIsMem,
                        String otherFilter,
                        String recomputeOtherFields,
                        String processName,
                        Map<String, String> processParams,
                        String resultLayerName,
                        String taskId) {

        log.info("进入通用叠加流程, 参数为 => baseFilter:" + baseFilter +
                ", recomputeBaseFields:" + recomputeBaseFields +
                ", otherFilter:" + otherFilter +
                ", recomputeOtherFields:" + recomputeOtherFields +
                ", processName:" + processName +
                ", processParams:" + toJSON(processParams));

        // 获取原始图层的信息
        RpcDescDataRespond baseLayerInfo = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(baseLayerName), taskId, UUIDUtils.getUUID());
        Map<String, String> baseLayerDescMap = baseLayerInfo.getDataDescMap();

        if (StringUtils.isBlank(baseLayerDescMap.get(constants.dataDescDefaultGeometry()))) {
            throw new NotSupportException("不支持非空间图层进行空间计算操作, 图层元数据信息为：" + toJSON(baseLayerDescMap), Thread.currentThread(), 3);
        }

        // 获取叠加图层的信息
        RpcDescDataRespond otherLayerInfo = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(otherLayerName), taskId, UUIDUtils.getUUID());

        Pair<String, String> filterLayerNames =
                overlayFilterPrepare(baseLayerName, baseIsMem, baseFilter, baseLayerInfo,
                otherLayerName, otherIsMem, otherFilter, otherLayerInfo, processName, taskId);

        String baseFilterLayer = filterLayerNames.getLeft();
        String otherFilterLayer = filterLayerNames.getRight();

        overlayExecute(baseFilterLayer, baseLayerDescMap, recomputeBaseFields,
                otherFilterLayer, recomputeOtherFields, processName, processParams,
                resultLayerName, taskId);
    }

    private Pair<String, String> overlayFilterPrepare(
            String baseLayerName,
            boolean baseIsMem,
            String baseFilter,
            RpcDescDataRespond baseLayerInfo,
            String otherLayerName,
            boolean otherIsMem,
            String otherFilter,
            RpcDescDataRespond otherLayerInfo,
            String processName,
            String taskId){

        String nBaseFilter = baseFilter;
        String nOtherFilter = otherFilter;
        if (StringUtils.equalsAnyIgnoreCase(processName,
                constants.clipProcessName(), constants.intersectionProcessName(),
                constants.eraseProcessName(), constants.identityProcessName())) {
            Map<Integer, String> filterInfos =
                    GeometryBoundaryUtils.getExtraFilter(baseLayerInfo, otherLayerInfo, OVERLAY_OPTIMIZE_RATE);
            if (StringUtils.equalsAnyIgnoreCase(processName, constants.clipProcessName(),
                    constants.intersectionProcessName())) {
                if (!baseIsMem) {
                    String baseSpatialFilter = filterInfos.getOrDefault(0, null);
                    log.info("基础图层新增筛选条件:" + baseSpatialFilter);
                    nBaseFilter = GeometryBoundaryUtils.addExtraFilter(baseFilter, baseSpatialFilter);
                }
            }
            if (!otherIsMem) {
                String otherSpatialFilter = filterInfos.getOrDefault(1, null);
                log.info("叠加图层新增筛选条件:" + otherSpatialFilter);
                nOtherFilter = GeometryBoundaryUtils.addExtraFilter(otherFilter, otherSpatialFilter);
            }
        }

        // 进行属性筛选
        String bFilterLayerName = filter(baseLayerName, nBaseFilter, taskId, UUIDUtils.getUUID());
        String oFilterLayerName = filter(otherLayerName, nOtherFilter, taskId, UUIDUtils.getUUID());

        log.info("属性筛选后的基础图层为:" + bFilterLayerName);
        log.info("属性筛选后的叠加图层为:" + oFilterLayerName);

        return new ImmutablePair<>(bFilterLayerName, oFilterLayerName);
    }


    private void overlayExecute(
            String bFilterLayerName,
            Map<String, String> baseLayerDescMap,
            String recomputeBaseFields,
            String oFilterLayerName,
            String recomputeOtherFields,
            String processName,
            Map<String, String> processParams,
            String resultLayerName,
            String taskId
    ){
        // 获取坐标转换后的图层
        String otherTransLayerName = transFormCrs(oFilterLayerName,
                baseLayerDescMap.get(constants.dataDescCrs()),
                taskId, UUIDUtils.getUUID());

        log.info("坐标转换后的叠加图层为:" + otherTransLayerName);

        // 面积计算
        ComputeLayerInfo bfComputerLayerInfo = computeArea(bFilterLayerName,
                StringUtils.isNotBlank(recomputeBaseFields), taskId, UUIDUtils.getUUID());
        ComputeLayerInfo ofComputerLayerInfo = computeArea(otherTransLayerName,
                StringUtils.isNotBlank(recomputeOtherFields), taskId, UUIDUtils.getUUID());

        log.info("面积计算后的基础图层为:" + bfComputerLayerInfo.getComputeLayerName());
        log.info("面积计算后基础图层追加的面积字段为:" + bfComputerLayerInfo.getAreaName());
        log.info("面积计算后的叠加图层为:" + ofComputerLayerInfo.getComputeLayerName());
        log.info("面积计算后叠加图层图层追加的面积字段为:" + ofComputerLayerInfo.getAreaName());

        // 获取执行结果
        String preResultLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_r_" + UUIDUtils.getUUID();
        processServer.binaryProcess(bfComputerLayerInfo.getComputeLayerName(),
                ofComputerLayerInfo.getComputeLayerName(), processName,
                processParams, preResultLayerName, taskId, UUIDUtils.getUUID());
        log.info("叠加执行结果图层为:" + preResultLayerName);

        if (StringUtils.isNotBlank(recomputeBaseFields) || StringUtils.isNotBlank(recomputeOtherFields)) {
            recomputeFields(bfComputerLayerInfo.getComputeLayerName(),
                    bfComputerLayerInfo.getAreaName(),
                    recomputeBaseFields,
                    ofComputerLayerInfo.getComputeLayerName(),
                    ofComputerLayerInfo.getAreaName(),
                    recomputeOtherFields,
                    preResultLayerName,
                    resultLayerName,
                    taskId);
        } else {
            dataServer.renameData(preResultLayerName, resultLayerName, taskId, UUIDUtils.getUUID());
        }
    }

    /**
     * 重算字段方法
     *
     * @param bLayerName           基础图层名称
     * @param bAreaName            基础图层面积字段名称
     * @param recomputeBaseFields  重算基础图层字段名称
     * @param oLayerName           其他图层名称
     * @param oAreaName            其他图层面积字段名称
     * @param recomputeOtherFields 重算其他图层字段名称
     * @param overlayLayerName     叠加过的图层名称
     * @param taskId               任务Id
     */
    public void recomputeFields(String bLayerName,
                                String bAreaName,
                                String recomputeBaseFields,
                                String oLayerName,
                                String oAreaName,
                                String recomputeOtherFields,
                                String overlayLayerName,
                                String resultLayerName,
                                String taskId) {
        // 获取计算面积后图层信息
        ComputeLayerInfo computerLayerInfo = computeArea(overlayLayerName,
                StringUtils.isNotBlank(recomputeBaseFields) || StringUtils.isNotBlank(recomputeOtherFields),
                taskId, UUIDUtils.getUUID());

        log.info("面积计算后的结果图层名称为:" + computerLayerInfo.getComputeLayerName());
        log.info("面积计算后的结果图层面积字段为:" + computerLayerInfo.getAreaName());

        // 获取重算字段在结果字段中的映射关系
        int[] startIndex = {1};
        int[] bAreaNameIndexFlag = {-1};
        int[] oAreaNameIndexFlag = {-1};
        Map<Integer, String> recomputeBaseFieldMap =
                getRenameMap(bLayerName, bAreaName, bAreaNameIndexFlag, recomputeBaseFields, taskId, StringUtils.isNotBlank(recomputeOtherFields), startIndex);
        Map<Integer, String> recomputeUnionFieldMap =
                getRenameMap(oLayerName, oAreaName, oAreaNameIndexFlag, recomputeOtherFields, taskId, false, startIndex);

        logInfo(recomputeBaseFieldMap);
        logInfo(recomputeUnionFieldMap);

        // 进行字段重算并赋值
        recomputeRunSQL(computerLayerInfo.getComputeLayerName(),
                bAreaNameIndexFlag[0],
                oAreaNameIndexFlag[0],
                computerLayerInfo.getAreaName(),
                recomputeBaseFieldMap,
                recomputeUnionFieldMap,
                resultLayerName,
                taskId);
    }

    /**
     * 通过sql实现字段赋值，新增字段，并删除新增的面积字段信息
     *
     * @param computeLayerName       结果图层名称
     * @param lAreaIndex             基础图层的面积字段索引
     * @param rAreaIndex             其他图层的面积字段索引
     * @param oAreaName              结果图层的面积字段
     * @param recomputeBaseFieldMap  重算基础图层字段在叠加结果图层中的映射
     * @param recomputeOtherFieldMap 重算其他图层字段在叠加结果图层中的映射
     * @param taskId                 任务Id
     */
    public void recomputeRunSQL(String computeLayerName,
                                int lAreaIndex,
                                int rAreaIndex,
                                String oAreaName,
                                Map<Integer, String> recomputeBaseFieldMap,
                                Map<Integer, String> recomputeOtherFieldMap,
                                String resultLayerName,
                                String taskId) {

        AtlasRpcDataTag computeDataDescTag = AtlasTagUtils.getAtlasRpcDataTag(computeLayerName);

        // 获取计算后图层字段信息
        RpcDescDataRespond computeDataDescRespond = client.getSparkRpcClientApi().descData(
                computeDataDescTag, taskId, UUIDUtils.getUUID());


        List<String> fieldNames =
                RpcDescRespondUtils.getVectorFieldInfo(computeDataDescRespond)
                        .stream()
                        .map(FieldInfo::getFieldName)
                        .collect(Collectors.toList());
        log.info("面积计算后结果图层的字段名称为:" + String.join(",", fieldNames));

        String computeLayerView = RpcNameUtils.getViewName(computeLayerName);
        // 构建执行的sql
        StringBuilder sqlBuilder = new StringBuilder();
        sqlBuilder.append("select ");
        for (int fieldIndex = 0; fieldIndex < fieldNames.size(); fieldIndex++) {
            if (fieldIndex == lAreaIndex || fieldIndex == rAreaIndex
                    || StringUtils.equals(fieldNames.get(fieldIndex), oAreaName)) {
                continue;
            }
            if (recomputeBaseFieldMap.containsKey(fieldIndex)) {
                appendComputeField(sqlBuilder, fieldNames.get(fieldIndex), fieldNames.get(lAreaIndex), oAreaName);
                continue;
            }

            if (recomputeOtherFieldMap.containsKey(fieldIndex)) {
                appendComputeField(sqlBuilder, fieldNames.get(fieldIndex), fieldNames.get(rAreaIndex), oAreaName);
                continue;
            }
            appendField(sqlBuilder, fieldNames.get(fieldIndex));
        }
        if (!recomputeBaseFieldMap.isEmpty()) {
            for (int rIndex : recomputeBaseFieldMap.keySet()) {
                renameRawComputeField(sqlBuilder, fieldNames, rIndex);
            }
        }

        if (!recomputeOtherFieldMap.isEmpty()) {
            for (int oIndex : recomputeOtherFieldMap.keySet()) {
                renameRawComputeField(sqlBuilder, fieldNames, oIndex);
            }
        }

        // 删除逗号
        sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
        sqlBuilder.append(" from `").append(computeLayerView).append("`");

        String sql = sqlBuilder.toString();

        Map<String, String> sqlParams = new HashMap<>();
        sqlParams.put(constants.runSqlExpression(), sql);
        sqlParams.put(constants.sqlStatisticLayerOrder(), "0");
        String message = client.getSparkRpcClientApi().listProcess(
                Collections.singletonList(AtlasTagUtils.getAtlasRpcDataTag(computeLayerName)),
                constants.runSqlProcessName(),
                sqlParams,
                AtlasTagUtils.getAtlasRpcDataTag(resultLayerName),
                taskId,
                UUIDUtils.getUUID());
        log.info(message);
    }

    public void renameRawComputeField(StringBuilder sqlBuilder,
                                      List<String> fieldNames,
                                      int fieldIndex) {
        String fieldName = fieldNames.get(fieldIndex);
        String bakField = fieldName + BAK_FIELD_SUFFIX;
        bakField = com.geoway.atlas.common.utils.StringUtils.getUniqueStringFromSeq(bakField, fieldNames);
        sqlBuilder.append("`").append(fieldName).append("` AS `").append(bakField).append("`,");
    }

    public void appendComputeField(StringBuilder sqlBuilder,
                                   String fieldName,
                                   String baseAreaName,
                                   String overAreaName) {

        sqlBuilder.append("`").append(fieldName).append("`*`")
                .append(overAreaName).append("`/`")
                .append(baseAreaName).append("` AS `")
                .append(fieldName).append("`,");

    }

    /**
     * 组件sql时增加字段
     *
     * @param sqlBuilder sql构建器
     * @param fieldName  字段名称
     */
    public void appendField(StringBuilder sqlBuilder, String fieldName) {
        sqlBuilder.append("`").append(fieldName).append("`,");
    }

    public Map<Integer, String> getRenameMap(String layerName,
                                             String areaName,
                                             int[] areaIndexFlag,
                                             String recomputeFields,
                                             String taskId,
                                             boolean addIndexFlag,
                                             int[] startIndexFlag) {

        if (StringUtils.isBlank(recomputeFields)) {
            // 如果重算字段为空且不增加索引值
            if (!addIndexFlag) {
                return new HashMap<>();
            }
            // 如果重算字段为空但增加索引值需要继续进行字段个数计算
        }

        // 获取当前图层的字段信息
        RpcDescDataRespond rpcDescDataRespond = client.getSparkRpcClientApi().descData(
                AtlasTagUtils.getAtlasRpcDataTag(layerName),
                taskId, UUIDUtils.getUUID());
        List<FieldInfo> fieldInfos = RpcDescRespondUtils.getVectorFieldInfo(rpcDescDataRespond);
        log.info("重算面积前字段为:" + fieldInfos.stream().map(FieldInfo::getFieldName).collect(Collectors.joining(",")));
        Map<String, String> fieldDescMap = rpcDescDataRespond.getDataDescMap();
        // 获取图斑字段名称
        String geometryField = fieldDescMap.get(constants.dataDescDefaultGeometry());
        if (StringUtils.isBlank(geometryField)) {
            throw new NotFoundException("不能找到图形字段!", Thread.currentThread(), 3);
        }

        // 如果重算字段为空则增加字段个数的信息
        if (StringUtils.isBlank(recomputeFields)) {
            startIndexFlag[0] += (fieldInfos.size() - 1);
            return new HashMap<>();
        }

        int startIndex = startIndexFlag[0];
        // 获取除了图形字段的索引信息
        List<String> fieldNames = fieldInfos.stream()
                .map(FieldInfo::getFieldName)
                .filter(fd -> !StringUtils.equals(fd, geometryField))
                .collect(Collectors.toList());
        // 重算图层字段名称在当前字段中的索引
        List<String> recomputeFieldList =
                Arrays.stream(recomputeFields.split(",")).map(String::trim).collect(Collectors.toList());
        if (!CollectionUtils.containsAll(fieldNames, recomputeFieldList)) {
            throw new ParamException("当前图层字段:" + StringUtils.join(fieldNames, ",")
                    + "不完全包含重算的图层字段:" + recomputeFields, Thread.currentThread(), 3);
        }
        Map<Integer, String> renameMap = new HashMap<>();
        recomputeFieldList.forEach(recomputeField ->
                renameMap.put(fieldNames.indexOf(recomputeField) + startIndex, recomputeField));
        areaIndexFlag[0] = fieldNames.indexOf(areaName) + startIndex;
        startIndexFlag[0] += fieldNames.size();

        return renameMap;

    }

    @SneakyThrows
    public LayerIdentityDto getLayerIdentityDto(String layerIdentity) {
        return objectMapper.readValue(layerIdentity, LayerIdentityDto.class);
    }

    public List<String> getIdentities(LayerIdentityDto layerIdentityDto) {
        return layerIdentityDto.getDatas().stream().map(this::toJSON).collect(Collectors.toList());
    }

    @SneakyThrows
    public String toJSON(Object obj) {
        return objectMapper.writeValueAsString(obj);
    }

    @SneakyThrows
    public <T> T fromJSON(String jsonString, Class<T> clz) {
        return objectMapper.readValue(jsonString, clz);
    }


    /**
     * 属性筛选
     *
     * @param layerName 图形名称
     * @param filter    筛选表达式
     * @param taskId    任务id
     * @param jobId     jobId
     * @return 返回筛选后的图层
     */
    public String filter(String layerName, String filter, String taskId, String jobId) {
        if (StringUtils.isNotBlank(filter)) {
            String tempLayerName = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_f_" + UUIDUtils.getUUID();
            Map<String, String> filterParams = new HashMap<>();
            filterParams.put(constants.filterExpression(), filter);
            String message =
                    processServer.filterProcess(layerName, filterParams, tempLayerName, taskId, jobId);
            log.info(message);
            return tempLayerName;
        } else {
            return layerName;
        }
    }

    /**
     * 计算面积
     *
     * @param layerName 图层名称
     * @param compute   是否计算
     * @param taskId    taskId
     * @param jobId     jobId
     * @return 返回计算面积后的图层名称
     */
    public ComputeLayerInfo computeArea(String layerName,
                                        boolean compute,
                                        String taskId,
                                        String jobId) {
        String areaName = AREA_FIELD_PREFIX + "_" + UUIDUtils.getUUID();
        return computeArea(layerName, compute, areaName, taskId, jobId);
    }

    /**
     * 计算面积
     *
     * @param layerName 图层名称
     * @param compute   是否计算
     * @param taskId    taskId
     * @param jobId     jobId
     * @return 返回计算面积后的图层名称
     */
    public ComputeLayerInfo computeArea(String layerName,
                                        boolean compute,
                                        String areaName,
                                        String taskId,
                                        String jobId) {
        if (!compute) {
            return new ComputeLayerInfo(layerName, null);
        } else {
            String resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_a_" + UUIDUtils.getUUID();
            Map<String, String> calculatorParams = new HashMap<String, String>();
            calculatorParams.put(constants.calculateFieldName(), areaName);
            calculatorParams.put(constants.calculateTarget(), constants.calculateTargetStArea());
            processServer.calculatorProcess(layerName, calculatorParams, resultLayer, taskId, jobId);
            return new ComputeLayerInfo(resultLayer, areaName);
        }
    }

    /**
     * 创建oid字段
     *
     * @param layerName 图层名称
     * @param taskId    taskId
     * @param jobId     jobId
     * @return 返回计算面积后的图层名称
     */
    public CreateOidLayerInfo createOid(String layerName,
                                        String taskId,
                                        String jobId) {

        Map<String, String> descMap = client.getSparkRpcClientApi()
                .descData(AtlasTagUtils.getAtlasRpcDataTag(layerName), taskId, jobId)
                .getDataDescMap();

        boolean delete = false;
        String oidFields = "";
        String resultLayer = layerName;
        if (descMap.containsKey(constants.dataDescOidFields())) {
            oidFields = descMap.get(constants.dataDescOidFields());
            if (oidFields.contains(",")) {
                log.warn("不支持图层包含多个oid,需要重建oid!");
                delete = true;
            }
        } else {
            delete = true;
        }
        if (delete) {
            oidFields = OID_FIELD_PREFIX + "_" + UUIDUtils.getUUID();
            resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_o_id_" + UUIDUtils.getUUID();
            Map<String, String> createOidParams = new HashMap<>();
            createOidParams.put(constants.createOidFieldName(), oidFields);
            createOidParams.put(constants.createOidLongField(), "true");
            client.getSparkRpcClientApi().unitaryProcess(AtlasTagUtils.getAtlasRpcDataTag(layerName),
                    constants.createOidProcessName(),
                    createOidParams,
                    AtlasTagUtils.getAtlasRpcDataTag(resultLayer),
                    taskId,
                    UUIDUtils.getUUID());
        }

        return new CreateOidLayerInfo(delete, oidFields, resultLayer);
    }

    /**
     * 转换到指定投影坐标系
     *
     * @param layerName 图层名称
     * @param crsString 坐标系参数
     * @param taskId    task的id
     * @param jobId     job的id
     * @return 返回转换后的图层名称
     */
    public String transFormCrs(String layerName,
                               String crsString,
                               String taskId,
                               String jobId) {
        String resultLayer = AtlasDataName.TEMP_LAYER_PREFIX() + "_" + UUIDUtils.getUUID();
        Map<String, String> transformCRS = new HashMap<>();
        transformCRS.put(constants.transformTargetCrs(), crsString);
        processServer.unitaryProcess(
                layerName,
                constants.transformCrsProcessName(),
                transformCRS,
                resultLayer,
                taskId,
                jobId);

        return resultLayer;
    }

    public void logInfo(Map<Integer, String> fieldMap) {
        if (fieldMap != null && !fieldMap.isEmpty()) {
            log.info("原始字段在结果字段中的映射为:" + toJSON(fieldMap));
        }
    }
}
