package com.geoway.atlas.web.api.v2.controller.pkg;

import com.geoway.atlas.web.api.v2.job.AsyncDeal;
import com.geoway.atlas.web.api.v2.service.pkg.GraphAnalysisServer;
import com.geoway.atlas.data.vector.spark.common.rpc.common.Constants;
import com.geoway.atlas.web.api.v2.utils.MapUtils;
import com.geoway.atlas.web.api.v2.utils.TaskUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.Map;

/**
 * @author zhaotong 2023/8/15 18:02
 */

@Api(value = "图谱分析组合算子", tags = "图谱分析组合算子")
@RequestMapping("/api/v2/package/analysis/graph")
@RestController
@CrossOrigin(
        origins = {"*"}
)
public class GraphAnalysisController {

    @Autowired
    private Constants constants;

    @Autowired
    private GraphAnalysisServer graphAnalysisServer;

    @ApiOperation("单源最短路径")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas_data_base_identity", value = "设置基础图层标识", required = true),
            @ApiImplicitParam(name = "atlas_process_bfs_source_filter", value = "设置源节点筛选条件", required = true),
            @ApiImplicitParam(name = "atlas_process_bfs_target_filter", value = "设置目标节点筛选条件", required = true),
            @ApiImplicitParam(name = "atlas_process_bfs_edge_filter", value = "设置边筛选条件"),
            @ApiImplicitParam(name = "atlas_process_bfs_max_path", value = "设置返回路径最多数量", defaultValue = "10"),
            @ApiImplicitParam(name = "atlas_process_bfs_edge_new_class", value = "新增边标识"),
            @ApiImplicitParam(name = "atlas_data_result_identity", value = "设置结果图层标识", required = true),
            @ApiImplicitParam(name = "atlas_data_result_persist_disk", value = "是否持久化存储"),
            @ApiImplicitParam(name = "atlas_data_result_persist_url", value = "持久化存储url"),
            @ApiImplicitParam(name = "atlas_storage_write_is_append", value = "是否追加写入"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @RequestMapping(value = {"/bfs"}, method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> graphIoProcess(
            @RequestParam(value = "atlas_data_base_identity") String baseIdentity,
            @RequestParam(value = "atlas_process_bfs_source_filter") String sourceFilter,
            @RequestParam(value = "atlas_process_bfs_target_filter") String targetFilter,
            @RequestParam(value = "atlas_process_bfs_edge_filter", required = false) String edgeFilter,
            @RequestParam(value = "atlas_process_bfs_max_path", required = false, defaultValue = "10") Integer maxPath,
            @RequestParam(value = "atlas_process_bfs_edge_new_class", required = false) String edgeNewClass,
            @RequestParam(value = "atlas_data_result_identity") String resultIdentity,
            @RequestParam(value = "atlas_data_result_persist_disk", required = false, defaultValue = "false") boolean persistDisk,
            @RequestParam(value = "atlas_data_result_persist_url", required = false) String persistUrl,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        // TODO 后面在filter中过滤
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps,
                Arrays.asList(
                        "atlas.data.base.identity",
                        "atlas.data.result.identity",
                        "atlas.data.result.persist.disk",
                        "atlas.data.result.persist.url",
                        "taskid",
                        "jobid"));
        return graphAnalysisServer.commonProcess(
                baseIdentity,
                constants.bfsProcessName(),
                paramMaps,
                resultIdentity,
                persistDisk,
                persistUrl,
                nTaskId,
                jobId,
                "完成计算单源最短路径");
    }

    @ApiOperation("最大连通子图")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas_data_base_identity", value = "设置基础图层标识", required = true),
            @ApiImplicitParam(name = "atlas_process_graph_connected_field", value = "设置连通分量字段", required = true),
            @ApiImplicitParam(name = "atlas_data_result_identity", value = "设置结果图层标识", required = true),
            @ApiImplicitParam(name = "atlas_data_result_persist_disk", value = "是否持久化存储"),
            @ApiImplicitParam(name = "atlas_data_result_persist_url", value = "持久化存储url"),
            @ApiImplicitParam(name = "atlas_storage_write_is_append", value = "是否追加写入"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @RequestMapping(value = {"/connected-graph"}, method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> connectedGraph(
            @RequestParam(value = "atlas_data_base_identity") String baseIdentity,
            @RequestParam(value = "atlas_process_graph_connected_field") String connectedField,
            @RequestParam(value = "atlas_data_result_identity") String resultIdentity,
            @RequestParam(value = "atlas_data_result_persist_disk", required = false, defaultValue = "false") boolean persistDisk,
            @RequestParam(value = "atlas_data_result_persist_url", required = false) String persistUrl,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        // TODO 后面在filter中过滤
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps,
                Arrays.asList(
                        "atlas.data.base.identity",
                        "atlas.data.result.identity",
                        "atlas.data.result.persist.disk",
                        "atlas.data.result.persist.url",
                        "taskid",
                        "jobid"));
        return graphAnalysisServer.commonProcess(
                baseIdentity,
                constants.connectedGraphProcessName(),
                paramMaps,
                resultIdentity,
                persistDisk,
                persistUrl,
                nTaskId,
                jobId,
                "完成计算最大连通子图");
    }

    @ApiOperation("标签传播算法")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas_data_base_identity", value = "设置基础图层标识", required = true),
            @ApiImplicitParam(name = "atlas_process_lpa_label_field", value = "设置标签字段", required = true),
            @ApiImplicitParam(name = "atlas_process_lpa_max_iter", value = "设置最大迭代次数"),
            @ApiImplicitParam(name = "atlas_data_result_identity", value = "设置结果图层标识", required = true),
            @ApiImplicitParam(name = "atlas_data_result_persist_disk", value = "是否持久化存储"),
            @ApiImplicitParam(name = "atlas_data_result_persist_url", value = "持久化存储url"),
            @ApiImplicitParam(name = "atlas_storage_write_is_append", value = "是否追加写入"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @RequestMapping(value = {"/label-propagation"}, method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> labelPropagation(
            @RequestParam(value = "atlas_data_base_identity") String baseIdentity,
            @RequestParam(value = "atlas_process_lpa_label_field") String labelField,
            @RequestParam(value = "atlas_process_lpa_max_iter", required = false, defaultValue = "30") Integer maxIter,
            @RequestParam(value = "atlas_data_result_identity") String resultIdentity,
            @RequestParam(value = "atlas_data_result_persist_disk", required = false, defaultValue = "false") boolean persistDisk,
            @RequestParam(value = "atlas_data_result_persist_url", required = false) String persistUrl,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        // TODO 后面在filter中过滤
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps,
                Arrays.asList(
                        "atlas.data.base.identity",
                        "atlas.data.result.identity",
                        "atlas.data.result.persist.disk",
                        "atlas.data.result.persist.url",
                        "taskid",
                        "jobid"));
        return graphAnalysisServer.commonProcess(
                baseIdentity,
                constants.labelPropagationProcessName(),
                paramMaps,
                resultIdentity,
                persistDisk,
                persistUrl,
                nTaskId,
                jobId,
                "完成标签传播算法!");
    }

    @ApiOperation("网页排名算法")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas_data_base_identity", value = "设置基础图层标识", required = true),
            @ApiImplicitParam(name = "atlas_process_pagerank_rank_field", value = "设置排名字段", required = true),
            @ApiImplicitParam(name = "atlas_process_pagerank_weight_field", value = "设置权重字段", required = true),
            @ApiImplicitParam(name = "atlas_process_pagerank_max_iter", value = "设置最大迭代次数"),
            @ApiImplicitParam(name = "atlas_data_result_identity", value = "设置结果图层标识", required = true),
            @ApiImplicitParam(name = "atlas_data_result_persist_disk", value = "是否持久化存储"),
            @ApiImplicitParam(name = "atlas_data_result_persist_url", value = "持久化存储url"),
            @ApiImplicitParam(name = "atlas_storage_write_is_append", value = "是否追加写入"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @RequestMapping(value = {"/page-rank"}, method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> pageRank(
            @RequestParam(value = "atlas_data_base_identity") String baseIdentity,
            @RequestParam(value = "atlas_process_pagerank_rank_field") String rankField,
            @RequestParam(value = "atlas_process_pagerank_weight_field") String weightField,
            @RequestParam(value = "atlas_process_pagerank_max_iter", required = false, defaultValue = "30") Integer maxIter,
            @RequestParam(value = "atlas_data_result_identity") String resultIdentity,
            @RequestParam(value = "atlas_data_result_persist_disk", required = false, defaultValue = "false") boolean persistDisk,
            @RequestParam(value = "atlas_data_result_persist_url", required = false) String persistUrl,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        // TODO 后面在filter中过滤
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps,
                Arrays.asList(
                        "atlas.data.base.identity",
                        "atlas.data.result.identity",
                        "atlas.data.result.persist.disk",
                        "atlas.data.result.persist.url",
                        "taskid",
                        "jobid"));
        return graphAnalysisServer.commonProcess(
                baseIdentity,
                constants.pageRankProcessName(),
                paramMaps,
                resultIdentity,
                persistDisk,
                persistUrl,
                nTaskId,
                jobId,
                "完成网页排名算法!");
    }

    @ApiOperation("计数三角形")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "atlas_data_base_identity", value = "设置基础图层标识", required = true),
            @ApiImplicitParam(name = "atlas_process_triangle_count_field", value = "设置计数字段", required = true),
            @ApiImplicitParam(name = "atlas_data_result_identity", value = "设置结果图层标识", required = true),
            @ApiImplicitParam(name = "atlas_data_result_persist_disk", value = "是否持久化存储"),
            @ApiImplicitParam(name = "atlas_data_result_persist_url", value = "持久化存储url"),
            @ApiImplicitParam(name = "atlas_storage_write_is_append", value = "是否追加写入"),
            @ApiImplicitParam(name = "taskid", defaultValue = "", value = "作业id"),
            @ApiImplicitParam(name = "jobid", defaultValue = "", value = "任务项id")
    })
    @RequestMapping(value = {"/triangle-count"}, method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_VALUE)
    @AsyncDeal
    public Map<String, Object> triangleCount(
            @RequestParam(value = "atlas_data_base_identity") String baseIdentity,
            @RequestParam(value = "atlas_process_triangle_count_field") String countField,
            @RequestParam(value = "atlas_data_result_identity") String resultIdentity,
            @RequestParam(value = "atlas_data_result_persist_disk", required = false, defaultValue = "false") boolean persistDisk,
            @RequestParam(value = "atlas_data_result_persist_url", required = false) String persistUrl,
            @RequestParam(value = "taskid", defaultValue = "", required = false) String taskId,
            @RequestParam(value = "jobid", defaultValue = "", required = false) String jobId,
            Map<String, String> paramMaps) {
        // TODO 后面在filter中过滤
        String nTaskId = TaskUtils.removeDelimiter(taskId);
        // 获取所有的输入参数
        MapUtils.filterKeys(paramMaps,
                Arrays.asList(
                        "atlas.data.base.identity",
                        "atlas.data.result.identity",
                        "atlas.data.result.persist.disk",
                        "atlas.data.result.persist.url",
                        "taskid",
                        "jobid"));
        return graphAnalysisServer.commonProcess(
                baseIdentity,
                constants.triangleCountProcessName(),
                paramMaps,
                resultIdentity,
                persistDisk,
                persistUrl,
                nTaskId,
                jobId,
                "完成计数三角形算法!");
    }


}
