package com.geoway.atlas.web.api.v2.pool;

import com.geoway.atlas.web.api.v2.component.bean.AtlasGisToolkitBeanFactory;
import com.geoway.atlas.web.api.v2.config.HikariPoolConfig;
import com.geoway.atlas.web.api.v2.exception.AtlasException;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.pool.HikariPool;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 数据库连接池管理
 *
 * @author zhaotong 2023/7/7 9:16
 */
public class DBPool {

    private static final String JDBC = "jdbc";

    /**
     * 1 - 数据库标识
     * 3 - 用户名
     * 4 - 密码
     * 5 - 主机+端口
     * 8 - 数据库名称
     * 9 - 数据库连接参数信息
     */
    private static final Pattern DB_PATTERN = Pattern.compile("^(.*?)://((.*?)/(.*)@)?(.*?)((/(.*?))(\\?.*|$))");
    private static final Logger log = LoggerFactory.getLogger(DBPool.class);

    private static final long CLOSE_POOL_TIMEOUT_MS = 120000L;

    /**
     * 数据库连接池
     */
    private static final ConcurrentMap<String, HikariPoolInfo> POOL = new ConcurrentHashMap<>();


    /**
     * 同statement.executeQuery方法
     *
     * @param dbInfo          数据库连接参数
     * @param sql             需要执行的sql
     * @param convertFunction resultSet转换为结果对象的方法
     * @param <T>             结果对象泛型
     * @return 返回结果对象
     */
    public static <T> T executeQuery(String dbInfo, String sql, Function<ResultSet, T> convertFunction) {
        return executeQuery(dbInfo, sql, convertFunction, null);
    }

    /**
     * 同statement.executeQuery方法
     *
     * @param dbInfo          数据库连接参数
     * @param sql             需要执行的sql
     * @param convertFunction resultSet转换为结果对象的方法
     * @param extraProperties 额外的数据库连接池配置
     * @param <T>             结果对象泛型
     * @return 返回结果对象
     */
    public static <T> T executeQuery(String dbInfo, String sql, Function<ResultSet, T> convertFunction, Map<Object, Object> extraProperties) {
        waitForRun(dbInfo);
        HikariPoolInfo hikariPoolInfo = POOL.computeIfAbsent(dbInfo, k ->
                new HikariPoolInfo((new HikariPool(getHikariConfig(k, extraProperties)))));
        hikariPoolInfo.startConnection();
        Connection connection = null;
        try {
            connection = hikariPoolInfo.getHikariPool().getConnection();
            Statement statement = connection.createStatement();
            ResultSet resultSet = statement.executeQuery(sql);
            return convertFunction.apply(resultSet);
        } catch (SQLException e) {
            log.error("执行:" + sql + ",失败:" + ExceptionUtils.getRootCauseMessage(e));
            if (connection != null) {
                try {
                    connection.rollback();
                } catch (SQLException ex) {
                    log.error("回滚失败!");
                }
            }
            throw new RuntimeException(e);
        } finally {
            hikariPoolInfo.endConnection();
            hikariPoolInfo.setLastAccessTime(System.currentTimeMillis());
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (SQLException e) {
                log.error("释放连接失败:" + ExceptionUtils.getRootCauseMessage(e));
            }
        }
    }

    /**
     * 同statement.execute方法
     *
     * @param dbInfo 数据库连接参数
     * @param sql    需要执行的sql
     * @return 如果第一个结果是ResultSet对象，则为true;如果是更新计数或没有结果，则为False
     */
    public static boolean execute(String dbInfo, String sql) {
        return execute(dbInfo, sql, null);
    }

    /**
     * 同statement.execute方法
     *
     * @param dbInfo          数据库连接参数
     * @param sql             需要执行的sql
     * @param extraProperties 额外的数据库连接池配置
     * @return 如果第一个结果是ResultSet对象，则为true;如果是更新计数或没有结果，则为False
     */
    public static boolean execute(String dbInfo, String sql, Map<Object, Object> extraProperties) {
        waitForRun(dbInfo);
        HikariPoolInfo hikariPoolInfo = POOL.computeIfAbsent(dbInfo, k ->
                new HikariPoolInfo((new HikariPool(getHikariConfig(k, extraProperties)))));
        hikariPoolInfo.startConnection();
        Connection connection = null;
        try {
            connection = hikariPoolInfo.getHikariPool().getConnection();
            Statement statement = connection.createStatement();
            return statement.execute(sql);
        } catch (SQLException e) {
            log.error("执行:" + sql + ",失败:" + ExceptionUtils.getRootCauseMessage(e));
            if (connection != null) {
                try {
                    connection.rollback();
                } catch (SQLException ex) {
                    log.error("回滚失败!");
                }
            }
            throw new RuntimeException(e);
        } finally {
            hikariPoolInfo.endConnection();
            hikariPoolInfo.setLastAccessTime(System.currentTimeMillis());
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (SQLException e) {
                log.error("释放连接失败:" + ExceptionUtils.getRootCauseMessage(e));
            }
        }
    }

    /**
     * 同statement.executeUpdate
     *
     * @param dbInfo 数据库连接参数
     * @param sql    需要执行的sql
     * @return (1) SQL数据操作语言(DML)语句的行数;(2)不返回任何结果的SQL语句的行数为0
     */
    public static int executeUpdate(String dbInfo, String sql) {
        return executeUpdate(dbInfo, sql, null);
    }

    /**
     * 同statement.executeUpdate
     *
     * @param dbInfo          数据库连接参数
     * @param sql             需要执行的sql
     * @param extraProperties 额外的数据库连接池配置
     * @return (1) SQL数据操作语言(DML)语句的行数;(2)不返回任何结果的SQL语句的行数为0
     */
    public static int executeUpdate(String dbInfo, String sql, Map<Object, Object> extraProperties) {
        waitForRun(dbInfo);
        HikariPoolInfo hikariPoolInfo = POOL.computeIfAbsent(dbInfo, k ->
                new HikariPoolInfo((new HikariPool(getHikariConfig(k, extraProperties)))));
        hikariPoolInfo.startConnection();
        Connection connection = null;
        try {
            connection = hikariPoolInfo.getHikariPool().getConnection();
            Statement statement = connection.createStatement();
            return statement.executeUpdate(sql);
        } catch (SQLException e) {
            log.error("执行:" + sql + ",失败:" + ExceptionUtils.getRootCauseMessage(e));
            if (connection != null) {
                try {
                    connection.rollback();
                } catch (SQLException ex) {
                    log.error("回滚失败!");
                }
            }
            throw new RuntimeException(e);
        } finally {
            hikariPoolInfo.endConnection();
            hikariPoolInfo.setLastAccessTime(System.currentTimeMillis());
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (SQLException e) {
                log.error("释放连接失败:" + ExceptionUtils.getRootCauseMessage(e));
            }
        }
    }

    public synchronized static void removeIfNotUse() {
        if (!POOL.isEmpty()) {
            for (String hikariPoolKey : POOL.keySet()) {
                HikariPoolInfo hikariPoolInfo = POOL.get(hikariPoolKey);
                // 计数为0表示当前没有执行的任务
                if (hikariPoolInfo.getConnectionCount() == 0 &&
                        (System.currentTimeMillis() - hikariPoolInfo.getLastAccessTime()) > CLOSE_POOL_TIMEOUT_MS) {
                    hikariPoolInfo.getReentrantLock().lock();
                    try {
                        if (POOL.containsKey(hikariPoolKey)) {
                            hikariPoolInfo.setCleaning(true);
                            try {
                                hikariPoolInfo.getHikariPool().shutdown();
                            } catch (InterruptedException e) {
                                throw new RuntimeException(e);
                            }

                            POOL.remove(hikariPoolKey);
                            hikariPoolInfo.setCleaning(false);
                            hikariPoolInfo.getCondition().signalAll();
                        }
                    } finally {
                        hikariPoolInfo.getReentrantLock().unlock();
                    }

                }
            }
        }
    }

    /**
     * 判断等待可以执行
     *
     * @param dbInfo 数据库信息
     */
    public static void waitForRun(String dbInfo) {
        if (POOL.containsKey(dbInfo)) {
            HikariPoolInfo hikariPoolInfo = null;
            // 如果hikariPoolInfo被删除了不处理
            try {
                hikariPoolInfo = POOL.get(dbInfo);
            } catch (NullPointerException ignored) {
            }
            if (hikariPoolInfo != null) {
                // 如果正在清理连接池则等待
                if (hikariPoolInfo.isCleaning()) {
                    hikariPoolInfo.getReentrantLock().lock();
                    try {
                        while (hikariPoolInfo.isCleaning()) {
                            hikariPoolInfo.getCondition().await();
                        }
                    } catch (InterruptedException ignored) {
                    } finally {
                        hikariPoolInfo.getReentrantLock().unlock();
                    }
                }
            }
        }
    }

    /**
     * 获取连接池配置
     *
     * @param dbInfo          数据库连接参数
     * @param extraProperties 额外的数据库配置
     * @return 返回连接池配置
     */
    private static HikariConfig getHikariConfig(String dbInfo, Map<Object, Object> extraProperties) {
        String dbMode = StringUtils.substringBefore(dbInfo, ":");
        Map<String, String> connectInfo = new HashMap<>();
        if (JDBC.equalsIgnoreCase(dbMode)) {
            String dbUrl = StringUtils.substringAfter(dbInfo, ":");
            Matcher matcher = DB_PATTERN.matcher(dbUrl);
            if (matcher.find()) {
                if (matcher.groupCount() == 9) {
                    DBDialect dbDialect = AtlasGisToolkitBeanFactory.getBean(matcher.group(1).toLowerCase(), DBDialect.class);
                    String jdbcUrl = dbDialect.getJdbcUrl(matcher.group(5), matcher.group(8)) + matcher.group(9);
                    connectInfo.put("driverClassName", dbDialect.getDriverClassName());
                    connectInfo.put("jdbcUrl", jdbcUrl);
                    connectInfo.put("username", matcher.group(3));
                    connectInfo.put("password", matcher.group(4));
                } else {
                    throw new AtlasException("输入的路径不规范：" + dbInfo);
                }
            } else {
                throw new AtlasException("输入的路径不规范：" + dbInfo);
            }
        } else {
            throw new AtlasException("不支持当前的数据连接参数标识：" + dbMode);
        }

        Properties hikariProperties = new Properties();
        HikariPoolConfig hikariPoolConfig = AtlasGisToolkitBeanFactory.getBean(HikariPoolConfig.class);
        Map hikariMap = AtlasGisToolkitBeanFactory.getObjectMapper().convertValue(hikariPoolConfig, Map.class);
        hikariProperties.putAll(hikariMap);
        hikariProperties.putAll(connectInfo);
        if (extraProperties != null) {
            hikariProperties.putAll(extraProperties);
        }
        HikariConfig hikariConfig = new HikariConfig(hikariProperties);
        return hikariConfig;
    }
}
