/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.nature;

import java.awt.geom.Point2D;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.TimeZone;
import org.geotools.nature.Calendar;

public class SunRelativePosition {
    private static final int DAY_MILLIS = 86400000;
    private static final double DARK = Double.NaN;
    public static final double ASTRONOMICAL_TWILIGHT = -18.0;
    public static final double NAUTICAL_TWILIGHT = -12.0;
    public static final double CIVIL_TWILIGHT = -6.0;
    private double twilight = -6.0;
    private long noonTime;
    private double azimuth;
    private double elevation;
    private double latitude;
    private double longitude;
    private long time = System.currentTimeMillis();
    private boolean updated;

    private static double sunEquationOfCenter(double t) {
        double m = Math.toRadians(SunRelativePosition.sunGeometricMeanAnomaly(t));
        return Math.sin(1.0 * m) * (1.914602 - t * (0.004817 + 1.4E-5 * t)) + Math.sin(2.0 * m) * (0.019993 - t * 1.01E-4) + Math.sin(3.0 * m) * 2.89E-4;
    }

    private static double sunGeometricMeanLongitude(double t) {
        double L0 = 280.46646 + t * (36000.76983 + 3.032E-4 * t);
        L0 -= 360.0 * Math.floor(L0 / 360.0);
        return L0;
    }

    private static double sunTrueLongitude(double t) {
        return SunRelativePosition.sunGeometricMeanLongitude(t) + SunRelativePosition.sunEquationOfCenter(t);
    }

    private static double sunApparentLongitude(double t) {
        double omega = Math.toRadians(125.04 - 1934.136 * t);
        return SunRelativePosition.sunTrueLongitude(t) - 0.00569 - 0.00478 * Math.sin(omega);
    }

    private static double sunGeometricMeanAnomaly(double t) {
        return 357.52911 + t * (35999.05029 - 1.537E-4 * t);
    }

    private static double eccentricityEarthOrbit(double t) {
        return 0.016708634 - t * (4.2037E-5 + 1.267E-7 * t);
    }

    private static double meanObliquityOfEcliptic(double t) {
        double seconds = 21.448 - t * (46.815 + t * (5.9E-4 - t * 0.001813));
        return 23.0 + (26.0 + seconds / 60.0) / 60.0;
    }

    private static double obliquityCorrected(double t) {
        double e0 = SunRelativePosition.meanObliquityOfEcliptic(t);
        double omega = Math.toRadians(125.04 - 1934.136 * t);
        return e0 + 0.00256 * Math.cos(omega);
    }

    private static double sunDeclination(double t) {
        double e = Math.toRadians(SunRelativePosition.obliquityCorrected(t));
        double b = Math.toRadians(SunRelativePosition.sunApparentLongitude(t));
        double sint = Math.sin(e) * Math.sin(b);
        double theta = Math.asin(sint);
        return Math.toDegrees(theta);
    }

    private static double solarNoonTime(double lon, double eqTime) {
        return 720.0 + lon * 4.0 - eqTime;
    }

    private static double equationOfTime(double t) {
        double eps = Math.toRadians(SunRelativePosition.obliquityCorrected(t));
        double l0 = Math.toRadians(SunRelativePosition.sunGeometricMeanLongitude(t));
        double m = Math.toRadians(SunRelativePosition.sunGeometricMeanAnomaly(t));
        double e = SunRelativePosition.eccentricityEarthOrbit(t);
        double y = Math.tan(eps / 2.0);
        y *= y;
        double sin2l0 = Math.sin(2.0 * l0);
        double cos2l0 = Math.cos(2.0 * l0);
        double sin4l0 = Math.sin(4.0 * l0);
        double sin1m = Math.sin(m);
        double sin2m = Math.sin(2.0 * m);
        double etime = y * sin2l0 - 2.0 * e * sin1m + 4.0 * e * y * sin1m * cos2l0 - 0.5 * y * y * sin4l0 - 1.25 * e * e * sin2m;
        return Math.toDegrees(etime) * 4.0;
    }

    private static double refractionCorrection(double zenith) {
        double exoatmElevation = 90.0 - zenith;
        if (exoatmElevation > 85.0) {
            return 0.0;
        }
        double te = Math.tan(Math.toRadians(exoatmElevation));
        double refractionCorrection = exoatmElevation > 5.0 ? 58.1 / te - 0.07 / (te * te * te) + 8.6E-5 / (te * te * te * te * te) : (exoatmElevation > -0.575 ? SunRelativePosition.getRefractionCorrectedElevation(exoatmElevation) : -20.774 / te);
        return refractionCorrection / 3600.0;
    }

    private static double getRefractionCorrectedElevation(double e) {
        return 1735.0 + e * (-518.2 + e * (103.4 + e * (-12.79 + e * 0.711)));
    }

    public SunRelativePosition() {
    }

    public SunRelativePosition(double twilight) throws IllegalArgumentException {
        this.setTwilight(twilight);
    }

    private void compute() {
        double latitude = this.latitude;
        double longitude = this.longitude;
        longitude = -longitude;
        double julianDay = Calendar.julianDay(this.time);
        double time = (julianDay - 2451545.0) / 36525.0;
        double solarDec = SunRelativePosition.sunDeclination(time);
        double eqTime = SunRelativePosition.equationOfTime(time);
        this.noonTime = Math.round(SunRelativePosition.solarNoonTime(longitude, eqTime) * 60000.0) + this.time / 86400000L * 86400000L;
        double trueSolarTime = (julianDay + 0.5 - Math.floor(julianDay + 0.5)) * 1440.0;
        trueSolarTime += eqTime - 4.0 * longitude;
        trueSolarTime -= 1440.0 * Math.floor(trueSolarTime / 1440.0);
        latitude = Math.toRadians(latitude);
        solarDec = Math.toRadians(solarDec);
        double csz = Math.sin(latitude) * Math.sin(solarDec) + Math.cos(latitude) * Math.cos(solarDec) * Math.cos(Math.toRadians(trueSolarTime / 4.0 - 180.0));
        if (csz > 1.0) {
            csz = 1.0;
        }
        if (csz < -1.0) {
            csz = -1.0;
        }
        double zenith = Math.acos(csz);
        double azDenom = Math.cos(latitude) * Math.sin(zenith);
        if (Math.abs(azDenom) > 0.001) {
            double azRad = (Math.sin(latitude) * Math.cos(zenith) - Math.sin(solarDec)) / azDenom;
            if (azRad > 1.0) {
                azRad = 1.0;
            }
            if (azRad < -1.0) {
                azRad = -1.0;
            }
            this.azimuth = 180.0 - Math.toDegrees(Math.acos(azRad));
            if (trueSolarTime > 720.0) {
                this.azimuth = -this.azimuth;
            }
        } else {
            this.azimuth = latitude > 0.0 ? 180.0 : 0.0;
        }
        this.azimuth -= 360.0 * Math.floor(this.azimuth / 360.0);
        double refractionCorrection = SunRelativePosition.refractionCorrection(Math.toDegrees(zenith));
        double solarZen = Math.toDegrees(zenith) - refractionCorrection;
        this.elevation = 90.0 - solarZen;
        if (this.elevation < this.twilight) {
            this.azimuth = Double.NaN;
            this.elevation = Double.NaN;
        }
        this.updated = true;
    }

    public void setCoordinate(double longitude, double latitude) {
        if (latitude > 89.8) {
            latitude = 89.8;
        }
        if (latitude < -89.8) {
            latitude = -89.8;
        }
        if (latitude != this.latitude || longitude != this.longitude) {
            this.latitude = latitude;
            this.longitude = longitude;
            this.updated = false;
        }
    }

    public void setCoordinate(Point2D point) {
        this.setCoordinate(point.getX(), point.getY());
    }

    public Point2D getCoordinate() {
        return new Point2D.Double(this.longitude, this.latitude);
    }

    public void setDate(Date date) {
        long time = date.getTime();
        if (time != this.time) {
            this.time = time;
            this.updated = false;
        }
    }

    public Date getDate() {
        return new Date(this.time);
    }

    public void setTwilight(double twilight) throws IllegalArgumentException {
        if (twilight < -90.0 || twilight > -90.0) {
            throw new IllegalArgumentException(String.valueOf(twilight));
        }
        this.twilight = twilight;
        this.updated = false;
    }

    public double getTwilight() {
        return this.twilight;
    }

    public double getAzimuth() {
        if (!this.updated) {
            this.compute();
        }
        return this.azimuth;
    }

    public double getElevation() {
        if (!this.updated) {
            this.compute();
        }
        return this.elevation;
    }

    public long getNoonTime() {
        if (!this.updated) {
            this.compute();
        }
        return this.noonTime % 86400000L;
    }

    public Date getNoonDate() {
        if (!this.updated) {
            this.compute();
        }
        return new Date(this.noonTime);
    }

    public static void main(String ... args) throws ParseException {
        DateFormat format = DateFormat.getDateTimeInstance(3, 3);
        format.setTimeZone(TimeZone.getTimeZone("UTC"));
        double longitude = 0.0;
        double latitude = 0.0;
        Date time = new Date();
        switch (args.length) {
            case 3: {
                time = format.parse(args[2]);
            }
            case 2: {
                latitude = Double.parseDouble(args[1]);
            }
            case 1: {
                longitude = Double.parseDouble(args[0]);
            }
        }
        SunRelativePosition calculator = new SunRelativePosition();
        calculator.setDate(time);
        calculator.setCoordinate(longitude, latitude);
        System.out.print("Date (UTC): ");
        System.out.println(format.format(time));
        System.out.print("Longitude:  ");
        System.out.println(longitude);
        System.out.print("Latitude:   ");
        System.out.println(latitude);
        System.out.print("Elevation:  ");
        System.out.println(calculator.getElevation());
        System.out.print("Azimuth:    ");
        System.out.println(calculator.getAzimuth());
        System.out.print("Noon date:  ");
        System.out.println(format.format(calculator.getNoonDate()));
    }
}

