package com.geoway.jckj.base.exception.handler;

import com.geoway.jckj.base.exception.login.NoLoginException;
import com.geoway.jckj.base.support.StringUtils;
import com.geoway.jckj.base.base.dto.BaseResponse;
import com.geoway.jckj.base.exception.ServiceException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletRequest;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author ALMJ
 * @desc 全局异常处理器
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {
    /**
     * 权限校验异常
     */
    @ExceptionHandler(AccessDeniedException.class)
    public BaseResponse handleAccessDeniedException(AccessDeniedException e, HttpServletRequest request) {
        String requestURI = request.getRequestURI();
        log.error("请求地址'{}',权限校验失败'{}'", requestURI, e.getMessage());
        return BaseResponse.buildFailuaResponse("没有权限，请联系管理员授权", HttpStatus.FORBIDDEN.value());
    }

    /**
     * 请求方式不支持
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public BaseResponse handleHttpRequestMethodNotSupported(HttpRequestMethodNotSupportedException e,
                                                            HttpServletRequest request) {
        String requestURI = request.getRequestURI();
        log.error("请求地址'{}',不支持'{}'请求", requestURI, e.getMethod());
        return BaseResponse.buildFailuaResponse(e.getMessage());
    }

    /**
     * 业务异常
     */
    @ExceptionHandler(ServiceException.class)
    public BaseResponse handleServiceException(ServiceException e, HttpServletRequest request) {
        log.error(e.getMessage(), e);
        Integer code = e.getCode();
        return StringUtils.isNotNull(code) ? BaseResponse.buildFailuaResponse(e.getMessage(), code) : BaseResponse.buildFailuaResponse(e.getMessage());
    }

    /**
     * 拦截未知的运行时异常
     */
    @ExceptionHandler(RuntimeException.class)
    public BaseResponse handleRuntimeException(RuntimeException e, HttpServletRequest request) {
        String requestURI = request.getRequestURI();
        log.error("请求地址'{}',发生未知异常.", requestURI, e);
        return BaseResponse.buildFailuaResponse(e.getMessage());
    }

    /**
     * 系统异常
     */
    @ExceptionHandler(Exception.class)
    public BaseResponse handleException(Exception e, HttpServletRequest request) {
        String requestURI = request.getRequestURI();
        log.error("请求地址'{}',发生系统异常.", requestURI, e);
        return BaseResponse.buildFailuaResponse(e.getMessage());
    }

    /**
     * 自定义验证异常
     */
    @ExceptionHandler(BindException.class)
    public BaseResponse handleBindException(BindException e) {
        log.error("参数绑定失败", e);
        String message = e.getAllErrors().get(0).getDefaultMessage();
        return BaseResponse.buildFailuaResponse(message);
    }
    /**
     * 自定义验证异常
     */
    @ExceptionHandler(NoLoginException.class)
    public BaseResponse handleNoLoginException(NoLoginException e) {
        log.error("token过期或有误", e);
        return BaseResponse.buildNoLoginResponse();
    }

    /**
     * 自定义验证异常
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResponse handleMethodArgumentNotValidException(HttpServletRequest request,MethodArgumentNotValidException e) {
        log.error("请求的url为{}出现数据校验异常,异常信息为:", request.getRequestURI(), e);
        BindingResult bindingResult = e.getBindingResult();
        List<String> errorMsgList = new ArrayList();
        for (FieldError fieldError : bindingResult.getFieldErrors()) {
            errorMsgList.add(fieldError.getDefaultMessage());
        }
        String message = errorMsgList.toString();
        return BaseResponse.buildFailuaResponse(message);
    }

}
