package com.geoway.jckj.base.config;

import com.geoway.jckj.base.support.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.context.WebServerInitializedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.Query;
import javax.servlet.http.HttpServletRequest;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Set;

/**
 * @author ALMJ
 * @desc 获取项目的IP和端口
 * @date 2021/9/8 11:31
 */
@Component
@Slf4j
public class ServerConfig implements ApplicationListener<WebServerInitializedEvent> {

    @Value("${server.servlet.context-path}")
    private String contexPath;


    @Value("${server.port}")
    private String port;

    public String getPort() {
        return port;
    }


    /**
     * 获取应用目录，如:  ns-onemap
     *
     * @return
     */
    public String getContexPath() {
        return this.contexPath;
    }

    /**
     * 获取当前运行端口
     *
     * @return
     */
    public String getServerPort() {
        return serverPort;
    }

    private String serverPort;

    /**
     * 获取完整url地址，如: 127.0.0.1:8097/ns-onemap
     *
     * @return
     */
    public String getUrl() {
        InetAddress address = null;
        try {
            address = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }
        return "http://" + address.getHostAddress() + ":" + this.port + this.contexPath;
    }

    /**
     * 获取运行IP地址
     *
     * @return
     */
    public String getHost() {
        InetAddress address = null;
        try {
            address = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }
        return address.getHostAddress();
    }

    /**
     * 获取客户端IP
     *
     * @param request
     * @return 客户端ip地址
     */
    public String getRemoteHost(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return ip.equals("0:0:0:0:0:0:0:1") ? "127.0.0.1" : ip;
    }

    @Override
    public void onApplicationEvent(WebServerInitializedEvent event) {
        String serverPort = String.valueOf(event.getWebServer().getPort());
        if (StringUtils.isBlank(serverPort)) {
            serverPort = getTomcatPort();
        }
        this.serverPort = serverPort;
    }

    /**
     * 部署在tomcat时获取运行端口
     *
     * @return
     */
    private String getTomcatPort() {
        String port = "";
        Set<ObjectName> objectNames = null;
        try {
            MBeanServer beanServer = ManagementFactory.getPlatformMBeanServer();
            objectNames = beanServer.queryNames(
                    new ObjectName("*:type=Connector,*"),
                    Query.match(Query.attr("protocol"), Query.value("HTTP/1.1")));
            port = objectNames.iterator().next().getKeyProperty("port");
            log.info("当前tomcat的运行端口已获取成功！对应的端口为:{}", port);
        } catch (MalformedObjectNameException e) {
            e.printStackTrace();
            log.error(e.getMessage());
        }
        return port;
    }


}
