package com.geoway.jckj.biz.plugin;

import cn.hutool.core.io.FileUtil;
import com.geoway.jckj.base.support.spring.SpringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Repository;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import java.io.File;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

@Component
public class PluginImportService {
    @Value("${plugin.enable:true}")
    private boolean enable = true;
    @Value("${plugin.pluginPath:plugins}")
    private String pluginPath;
    private ApplicationContext applicationContext;// 获取ApplicationContext
    private DefaultListableBeanFactory defaultListableBeanFactory;// 获取bean工厂并转换为DefaultListableBeanFactory

    public void importPlugins(){
        initContext();
        if(!enable){
            return;
        }
        if(StringUtils.isEmpty(pluginPath)){
            pluginPath = "plugins";
        }
        String curProjectPath = System.getProperty("user.dir");
        String pluginAbsolutePath = curProjectPath + File.separator + pluginPath;
        File pluginFile = new File(pluginAbsolutePath);
        if (!pluginFile.isDirectory() || !pluginFile.exists()) {
            return;
        }

        File[] jarFiles = pluginFile.listFiles((dir, name) -> name.endsWith(".jar"));
        if (jarFiles.length == 0) {
            return;
        }
        System.out.println(" <---------------开始热加载jar包 ---------------> ");
        try{
            for(File jarFile:jarFiles){
                if(!loadJar(jarFile.getAbsolutePath())){
                    System.out.println("******jar包加载失败：" + jarFile.getAbsolutePath());
                    continue;
                }
                if(!registerBean(jarFile)){
                    System.out.println("******jar注册失败：" + jarFile.getAbsolutePath());
                }
                System.out.println("******jar热加载成功：" + jarFile.getAbsolutePath());
            }
            System.out.println(" <--------------- 热加载jar包结束 ---------------> ");
        } catch (Exception e){
            System.out.println(" <---------------热加载jar包异常 ---------------> ");
            e.printStackTrace();
        }
    }

    private void initContext(){
        if(applicationContext != null){
            return;
        }
        applicationContext = SpringUtils.getApplicationContext();
        //将applicationContext转换为ConfigurableApplicationContext
        ConfigurableApplicationContext configurableApplicationContext = (ConfigurableApplicationContext) applicationContext;
        defaultListableBeanFactory = (DefaultListableBeanFactory) configurableApplicationContext.getBeanFactory();
    }

    private boolean loadJar(String jarPath) {
        try {
            //获取classloader
            URLClassLoader classLoader = (URLClassLoader)Thread.currentThread().getContextClassLoader();
            // 获取 URLClassLoader addURL() 方法
            Method add = URLClassLoader.class.getDeclaredMethod("addURL", new Class[]{URL.class});
            // 设置可访问
            if (!add.isAccessible()) {
                add.setAccessible(true);
            }
            // 获取jar路径
            File jarFile = FileUtil.file(jarPath);
            // 执行addURL()方法
            add.invoke(classLoader, new Object[]{jarFile.toURI().toURL()});
            //成功返回true
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        // 如果出错则返回false
        return false;
    }

    private boolean registerBean(File file){
        try{
            JarFile jarFile = new JarFile(file);
            //解析jar包每一项
            Enumeration<JarEntry> en = jarFile.entries();
            while (en.hasMoreElements()) {
                JarEntry je = en.nextElement();
                String name = je.getName();
                //这里添加了路径扫描限制
                if (name.endsWith(".class")) {
                    String className = name.replace(".class", "").replaceAll("/", ".");
                    Class<?> clazz = Class.forName(className);
                    insertBean(clazz);
                }
            }
            return true;
        } catch (Exception e){
            e.printStackTrace();
            return false;
        }
    }

    private void insertBean(Class<?> c){
        if(isSpringBeanClass(c)){
            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(c);
            BeanDefinition beanDefinition = builder.getBeanDefinition();
            //避免重复注册
            if(!defaultListableBeanFactory.containsBean(c.getName())){
                defaultListableBeanFactory.registerBeanDefinition(c.getName(),beanDefinition);
            }

        }
    }

    private boolean isSpringBeanClass(Class<?> cla){
        if(cla==null){
            return false;
        }
        //是否是接口
        if(cla.isInterface()){
            return false;
        }
        //是否是抽象类
        if( Modifier.isAbstract(cla.getModifiers())){
            return false;
        }
        if(cla.getAnnotation(Component.class)!=null){
            return true;
        }
        if(cla.getAnnotation(Repository.class)!=null){
            return true;
        }
        if(cla.getAnnotation(Service.class)!=null){
            return true;
        }
        return false;
    }
}
