package com.geoway.jckj.biz.service.oauth2.impl;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import com.geoway.jckj.base.exception.ServiceException;
import com.geoway.jckj.base.exception.login.NoLoginException;
import com.geoway.jckj.base.support.encryption.AESUtils;
import com.geoway.jckj.biz.entity.SysUser;
import com.geoway.jckj.biz.service.dev.unity.IUnityLoginService;
import com.geoway.jckj.biz.service.login.ISsoAppService;
import com.geoway.jckj.biz.service.oauth2.IOauth2Service;
import com.geoway.jckj.biz.service.sys.SysUserLimitService;

import com.geoway.jckj.biz.util.GuestUtil;
import com.geoway.jckj.biz.util.LoginUtil;
import com.geoway.jckj.biz.util.SysUserUtil;
import com.geoway.sso.client.enums.GrantTypeEnum;
import com.geoway.sso.client.enums.RoleLevelEnum;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.RpcAccessToken;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.CookieUtils;
import com.geoway.sso.server.common.AccessTokenContent;
import com.geoway.sso.server.common.CodeContent;
import com.geoway.sso.server.common.RefreshTokenContent;
import com.geoway.sso.server.config.PropertyConfig;
import com.geoway.sso.server.constant.AppConstant;
import com.geoway.sso.server.session.AccessTokenManager;
import com.geoway.sso.server.session.CodeManager;
import com.geoway.sso.server.session.RefreshTokenManager;
import com.geoway.sso.server.session.TicketGrantingTicketManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Date;

/**
 * @author 连世忠
 * @ClassName SsoOauth2ServiceImpl
 * @Description TODO
 * @date 2024/3/12 17:24
 * @Version 1.0
 */
@Service
public class Oauth2ServiceImpl implements IOauth2Service {
    @Autowired
    private ISsoAppService ssoAppService;

    @Autowired
    private IUnityLoginService userService;

    @Autowired
    private CodeManager codeManager;

    @Autowired
    private AccessTokenManager accessTokenManager;

    @Autowired
    private RefreshTokenManager refreshTokenManager;

    @Autowired
    private TicketGrantingTicketManager ticketGrantingTicketManager;

    @Autowired
    private SysUserLimitService sysUserLimitService;

    @Override
    public RpcAccessToken getAccessToken(String grantType, String appId, String appSecret, String code, String username, String password,String tel,String sessionId,String catpcha) {
         return getAccessToken( grantType,  appId,  appSecret,  code,  username,  password, tel, sessionId, catpcha,"","","","","");
    }

    @Override
    public RpcAccessToken getAccessToken(String grantType, String appId, String appSecret, String code, String username, String password, String tel, String sessionId, String catpcha, String caUserCert, String caOauthName,String oneProcessId, String oneToken, String oneAuthCode) {
        Result<Void> result = validateParam(grantType, code, username, password,tel,sessionId,catpcha,caUserCert,caOauthName,oneProcessId,oneToken,oneAuthCode);
        if (!result.isSuccess()) {
            throw new ServiceException(result.getMessage());
        }

        // 校验应用
        Result<Void> appResult = ssoAppService.validate(appId, appSecret);
        if (!appResult.isSuccess()) {
            throw new ServiceException(appResult.getMessage());
        }

        // 校验授权
        Result<AccessTokenContent> accessTokenResult = validateAuth(grantType, code, username, password,tel, appId,sessionId,catpcha,caUserCert,caOauthName,oneProcessId,oneToken,oneAuthCode);
        if (!accessTokenResult.isSuccess()) {
            throw new ServiceException(accessTokenResult.getMessage());
        }

        //检验是否加入黑名单
        boolean isValid = sysUserLimitService.isValid(appId,accessTokenResult.getData().getUser().getId());
        if (!isValid) {
            throw new ServiceException("该用户被限制访问，请联系管理员");
        }

        // 生成RpcAccessToken返回
        return LoginUtil.genereateRpcAccessToken(accessTokenResult.getData(), null);
    }

    @Override
    public RpcAccessToken refreshToken(String appId, String refreshToken) {
        if(GuestUtil.validateGuestToken(refreshToken)){
            return GuestUtil.getGuestRpcAccessToken(refreshToken);
        }
        if (!ssoAppService.exists(appId)) {
            throw new ServiceException("非法应用");
        }

        RefreshTokenContent refreshTokenContent = refreshTokenManager.validate(refreshToken);
        if (refreshTokenContent == null) {
            throw new NoLoginException("refreshToken有误或已过期");
        }
        AccessTokenContent accessTokenContent = refreshTokenContent.getAccessTokenContent();
        if (!appId.equals(accessTokenContent.getAppId())) {
            throw new ServiceException("非法应用");
        }
        SsoUser user = ticketGrantingTicketManager.getAndRefresh(accessTokenContent.getCodeContent().getTgt());
        if (user == null) {
            throw new NoLoginException("服务端session已过期");
        }
        return LoginUtil.genereateRpcAccessToken(accessTokenContent, refreshTokenContent.getAccessToken());
    }

    @Override
    public RpcAccessToken queryAccessToken(String token) {
        if(GuestUtil.validateGuestToken(token)){
            return GuestUtil.getGuestRpcAccessToken(token);
        }
        AccessTokenContent tokenContent = accessTokenManager.get(token);
        if (tokenContent != null) {
            RpcAccessToken rpcAccessToken = new RpcAccessToken(token, accessTokenManager.getExpiresIn(), tokenContent.getRefreshToken(), tokenContent.getUser());
            return rpcAccessToken;
        }
        return null;
    }

    @Override
    public void revokeAccessToken(String token) {
        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
        HttpServletResponse response = servletRequestAttributes.getResponse();
        AccessTokenContent tokenContent = accessTokenManager.get(token);
        if (tokenContent != null) {
            String tgt = tokenContent.getCodeContent().getTgt();
            // 删除登录凭证
            ticketGrantingTicketManager.remove(tgt);
            // 删除凭证Cookie
            CookieUtils.removeCookie(AppConstant.TGC, "/", response);
            // 删除所有tgt对应的调用凭证，并通知客户端登出注销本地session
            accessTokenManager.remove(tgt);
        } else {
            throw new NoLoginException("token有误或已过期");
        }

    }

    @Override
    public RpcAccessToken getAccessToken(String appId, String appSecret, String username, String password, String accessToken) {
        // 校验应用
        Result<Void> appResult = ssoAppService.validate(appId, appSecret);
        if (!appResult.isSuccess()) {
            throw new ServiceException(appResult.getMessage());
        }

        // app通过此方式由客户端代理转发http请求到服务端获取accessToken
        Result<SsoUser> loginResult = userService.login(username,password);
        if (!loginResult.isSuccess()) {
            throw new ServiceException(loginResult.getMessage());
        }
        SsoUser user = loginResult.getData();
        user.setToken(accessToken);
        String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
        CodeContent codeContent = new CodeContent(tgt, false, null);
        AccessTokenContent accessTokenContent = new AccessTokenContent(codeContent, user, appId,null);

        //检验是否加入黑名单
        boolean isValid = sysUserLimitService.isValid(appId,accessTokenContent.getUser().getId());
        if (!isValid) {
            throw new ServiceException("该用户被限制访问，请联系管理员");
        }

        // 生成RpcAccessToken返回
        return LoginUtil.genereateRpcAccessToken(accessTokenContent,null);
    }

    @Override
    public String getAccessCode(String appId, String appSecret, String accessToken) {
        Result<Void> appResult = ssoAppService.validate(appId, appSecret);
        if (!appResult.isSuccess()) {
            throw new ServiceException(appResult.getMessage());
        }
        AccessTokenContent tokenContent = accessTokenManager.get(accessToken);
        if (tokenContent != null) {
            String tgt = tokenContent.getCodeContent().getTgt();
            String code = codeManager.generate(tgt, true, "");
            return code;
        } else {
            throw new NoLoginException("token有误或已过期");
        }
    }


    private Result<Void> validateParam(String grantType, String code,
                                       String username, String password,
                                       String tel,String sessionId,String catpcha,
                                       String caUserCert, String caOauthName,
                                       String oneProcessId, String oneToken, String oneAuthCode) {
        if (GrantTypeEnum.AUTHORIZATION_CODE.getValue().equals(grantType)) {
            if (StringUtils.isEmpty(code)) {
                return Result.createError("code不能为空");
            }
        } else if (GrantTypeEnum.PASSWORD.getValue().equals(grantType)) {
            if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
                return Result.createError("username和password不能为空");
            }
        } else if(GrantTypeEnum.PASSWORD_CAPTCHA.getValue().equals(grantType)){
            if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
                return Result.createError("username和password不能为空");
            }
            if (StringUtils.isEmpty(sessionId) || StringUtils.isEmpty(catpcha)) {
                return Result.createError("验证码不能为空");
            }
        }else if (GrantTypeEnum.TEL_CAPTCHA.getValue().equals(grantType)) {
            if (StringUtils.isEmpty(tel) || StringUtils.isEmpty(catpcha)) {
                return Result.createError("手机号和验证码不能为空");
            }
        }else if(GrantTypeEnum.CA.getValue().equals(grantType)){
            if (StringUtils.isEmpty(caUserCert)) {
                return Result.createError("CA证书不能为空");
            }
        }else if(GrantTypeEnum.ONE.getValue().equals(grantType)){
            if (StringUtils.isEmpty(oneProcessId) || StringUtils.isEmpty(oneToken)) {
                return Result.createError("oneProcessId和oneToken不能为空");
            }
        }
        else {
            return Result.createError("授权方式不支持");
        }
        return Result.success();
    }

    private Result<AccessTokenContent> validateAuth(String grantType, String code, String username, String password,String tel,
                                                    String appId,String sessionId,String catpcha,String caUserCert, String caOauthName,
                                                    String oneProcessId, String oneToken, String oneAuthCode) {
        AccessTokenContent authDto = null;
        if (GrantTypeEnum.AUTHORIZATION_CODE.getValue().equals(grantType)) {
            CodeContent codeContent = codeManager.getAndRemove(code);
            if (codeContent == null) {
                return Result.createError("code有误或已过期");
            }
            SsoUser user = ticketGrantingTicketManager.getAndRefresh(codeContent.getTgt());
            if (user == null) {
                return Result.createError("服务端session已过期");
            }
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        } else if (GrantTypeEnum.PASSWORD.getValue().equals(grantType)) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            Result<SsoUser> loginResult = userService.login(username,password,"","",request);
            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }else if(GrantTypeEnum.PASSWORD_CAPTCHA.getValue().equals(grantType)){
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            Result<SsoUser> loginResult = userService.login(username,password,sessionId,catpcha,request);
            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }else if(GrantTypeEnum.TEL_CAPTCHA.getValue().equals(grantType)){
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            Result<SsoUser> loginResult = userService.loginBySms(tel,sessionId,catpcha,request);
            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }else if(GrantTypeEnum.CA.getValue().equals(grantType)){
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            Result<SsoUser> loginResult = userService.loginByCa(caUserCert,caOauthName,request);
            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }
        else if(GrantTypeEnum.ONE.getValue().equals(grantType)){
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            Result<SsoUser> loginResult = userService.loginByOne(oneProcessId,oneToken,oneAuthCode,request);
            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }
        return Result.createSuccess(authDto);
    }
}
