package com.geoway.sso.server.session.redis;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.geoway.sso.client.constant.Oauth2Constant;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.HttpServletUtil;
import com.geoway.sso.server.config.PropertyConfig;
import com.geoway.sso.server.constant.AppConstant;
import com.geoway.sso.server.session.TicketGrantingTicketManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.concurrent.TimeUnit;

/**
 * @author ALMJ
 * @desc 分布式登录凭证管理
 */
@Component
@ConditionalOnProperty(name = "sso.session.manager", havingValue = "redis")
public class RedisTicketGrantingTicketManager implements TicketGrantingTicketManager {

    @Autowired
    private PropertyConfig propertyConfig;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    @Override
    public void create(String tgt, SsoUser user) {
        redisTemplate.opsForValue().set(AppConstant.REDIS_TOKEN_PREFIX + tgt, JSON.toJSONString(user), getExpiresIn(),
                TimeUnit.SECONDS);
    }

    @Override
    public SsoUser getAndRefresh(String tgt) {
        String user = redisTemplate.opsForValue().get(AppConstant.REDIS_TOKEN_PREFIX + tgt);
        if (StringUtils.isEmpty(user)) {
            return null;
        }
        redisTemplate.expire(AppConstant.REDIS_TOKEN_PREFIX + tgt, getExpiresIn(), TimeUnit.SECONDS);
        return JSONObject.parseObject(user, SsoUser.class);
    }

    @Override
    public void set(String tgt, SsoUser user) {
        create(tgt, user);
    }

    @Override
    public void remove(String tgt) {
        redisTemplate.delete(AppConstant.REDIS_TOKEN_PREFIX + tgt);
    }

    @Override
    public int getExpiresIn() {
        HttpServletRequest request = HttpServletUtil.getRequest();
        if (request != null) {
            if (ObjectUtil.equal(request.getParameter(Oauth2Constant.APP_FLAG), "true")) {
                return propertyConfig.getSsoAppTokenTimeout();
            }
        }
        return propertyConfig.getSsoTimeout();
    }
}