package com.gw.base.aop;
import com.gw.base.lang.invoke.GaMethodHandDefine;
import com.gw.base.lang.invoke.GkMethodHand;
import com.gw.base.util.GutilClass;
import com.gw.base.util.GutilReflection;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * 动态代理
 * @author Ray
 *
 */
public interface GiProxyAspect {


	/**
	 * 使用切面代理对象
	 *
	 * @param <T> 切面对象类型
	 * @param target 目标对象
	 * @param aspectClass 切面对象类
	 * @return 代理对象
	 */

	public static <T> T proxy(T target, Class<? extends GiProxyAspect> aspectClass){
		return (T)proxy(target,GutilReflection.newInstance(aspectClass));
	}

	/**
	 * 使用切面代理对象
	 *
	 * @param <T> 被代理对象类型
	 * @param target 被代理对象
	 * @param aspect 切面对象
	 * @return 代理对象
	 */
	@GaMethodHandDefine(expectClassName = "com.gw.spi.aop.DynamicProxy4Gw",expectMethodName = "proxy")
	public static <T> T proxy(T target, GiProxyAspect aspect){
		return (T)GkMethodHand.invokeSelf(target,aspect);
	}

	/**
	 * 创建动态代理对象<br>
	 * 动态代理对象的创建原理是：<br>
	 * 假设创建的代理对象名为 $Proxy0<br>
	 * 1、根据传入的interfaces动态生成一个类，实现interfaces中的接口<br>
	 * 2、通过传入的classloder将刚生成的类加载到jvm中。即将$Proxy0类load<br>
	 * 3、调用$Proxy0的$Proxy0(InvocationHandler)构造函数 创建$Proxy0的对象，并且用interfaces参数遍历其所有接口的方法，这些实现方法的实现本质上是通过反射调用被代理对象的方法<br>
	 * 4、将$Proxy0的实例返回给客户端。 <br>
	 * 5、当调用代理类的相应方法时，相当于调用 {@link InvocationHandler#invoke(Object, java.lang.reflect.Method, Object[])} 方法
	 *
	 *
	 * @param <T> 被代理对象类型
	 * @param classloader 被代理类对应的ClassLoader
	 * @param invocationHandler {@link InvocationHandler} ，被代理类通过实现此接口提供动态代理功能
	 * @param interfaces 代理类中需要实现的被代理类的接口方法
	 * @return 代理类
	 */
	@SuppressWarnings("unchecked")
	public static <T> T newProxyInstance(ClassLoader classloader, InvocationHandler invocationHandler, Class<?>... interfaces) {
		return (T) Proxy.newProxyInstance(classloader, interfaces, invocationHandler);
	}

	/**
	 * 创建动态代理对象
	 *
	 * @param <T> 被代理对象类型
	 * @param invocationHandler {@link InvocationHandler} ，被代理类通过实现此接口提供动态代理功能
	 * @param interfaces 代理类中需要实现的被代理类的接口方法
	 * @return 代理类
	 */
	public static <T> T newProxyInstance(InvocationHandler invocationHandler, Class<?>... interfaces) {
		return newProxyInstance(GutilClass.getDefaultClassLoader(), invocationHandler, interfaces);
	}



	/**
	 * 目标方法执行前的操作
	 *
	 * @param target 目标对象
	 * @param method 目标方法
	 * @param args   参数
	 * @return 是否继续执行接下来的操作
	 */
	boolean before(Object target, Method method, Object[] args);

	/**
	 *
	 * @param target    目标对象
	 * @param method    目标方法
	 * @param args      参数
	 * @param returnVal 目标方法执行返回值
	 * @return 是否允许返回值（接下来的操作）
	 */
	boolean after(Object target, Method method, Object[] args, Object returnVal);

	/**
	 * 目标方法抛出异常时的操作
	 *
	 * @param target 目标对象
	 * @param method 目标方法
	 * @param args   参数
	 * @param e      异常
	 * @return 是否允许抛出异常
	 */
	boolean afterException(Object target, Method method, Object[] args, Throwable e);


}
