package com.gw.base.gpa.support;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;
import com.gw.base.gpa.support.GwOrder.Direction;
import com.gw.base.util.GutilAssert;
import com.gw.base.util.GutilStr;


/**
 * Sort 排序条件对象
 */
public class GwSort implements Serializable {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = -1586646056446727841L;

	private static final GwSort UNSORTED = GwSort.by(new GwOrder[0]);

	public static final Direction DEFAULT_DIRECTION = Direction.ASC;

	private final List<GwOrder<?>> orders;

	public GwSort(List<GwOrder<?>> orders) {
		this.orders = orders;
	}

	/**
	 * Creates a new {@link GwSort} for the given {@link GwOrder}s.
	 *
	 * @param orders must not be {@literal null}.
	 * @return
	 */
	public static GwSort by(List<GwOrder<?>> orders) {

		GutilAssert.notNull(orders, "GwOrders must not be null!");

		return orders.isEmpty() ? GwSort.unsorted() : new GwSort(orders);
	}

	/**
	 * Creates a new {@link GwSort} for the given {@link GwOrder}s.
	 *
	 * @param orders must not be {@literal null}.
	 * @return
	 */
	public static GwSort by(GwOrder... orders) {

		GutilAssert.notNull(orders, "GwOrders must not be null!");

		return new GwSort(Arrays.asList(orders));
	}



	/**
	 * Returns a {@link GwSort} instances representing no sorting setup at all.
	 *
	 * @return
	 */
	public static GwSort unsorted() {
		return UNSORTED;
	}

	/**
	 * (non-Javadoc)
	 * @see java.lang.List#isEmpty()
	 */
	public boolean isEmpty() {
		return orders.isEmpty();
	}
	
	/**
	 * (non-Javadoc)
	 * @see java.lang.List#clear()
	 */
	public void clear() {
		this.orders.clear();
	}

	/**
	 * Returns a new {@link GwSort} consisting of the {@link GwOrder}s of the current {@link GwSort} combined with the given
	 * ones.
	 *
	 * @param sort must not be {@literal null}.
	 * @return
	 */
	public GwSort and(GwSort sort) {

		GutilAssert.notNull(sort, "Sort must not be null!");

		ArrayList<GwOrder<?>> these = new ArrayList<>();

		for (GwOrder<?> order : orders) {
			these.add(order);
		}

		return GwSort.by(these);
	}

	/**
	 * Returns the order registered for the given property.
	 *
	 * @param property
	 * @return
	 */
	public GwOrder<?> getOrderFor(String property) {

		for (GwOrder<?> order : orders) {
			if (order.getProperty().equals(property)) {
				return order;
			}
		}

		return null;
	}

	/**
	 * (non-Javadoc)
	 * @see java.lang.Iterable#iterator()
	 */
	public Iterator<GwOrder<?>> iterator() {
		return this.orders.iterator();
	}
	
	
	/**
	 * (non-Javadoc)
	 * @see java.lang.List#forEach()
	 */
	public void forEach(Consumer<? super GwOrder> action) {
		this.orders.forEach(action);
	}
	
	/**
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {

		if (this == obj) {
			return true;
		}

		if (!(obj instanceof GwSort)) {
			return false;
		}

		GwSort that = (GwSort) obj;

		return orders.equals(that.orders);
	}

	/**
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {

		int result = 17;
		result = 31 * result + orders.hashCode();
		return result;
	}

	/**
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return isEmpty() ? "UNSORTED" : GutilStr.collectionToCommaDelimitedString(orders);
	}





}
