package com.gw.base.concurrent;


abstract public class GkUnsynchronizedAppenderBase<E>{

    protected boolean started = false;

    // using a ThreadLocal instead of a boolean add 75 nanoseconds per
    // doAppend invocation. This is tolerable as doAppend takes at least a few microseconds
    // on a real appender
    /**
     * The guard prevents an appender from repeatedly calling its own doAppend
     * method.
     */
    private ThreadLocal<Boolean> guard = new ThreadLocal<Boolean>();

    /**
     * Appenders are named.
     */
    protected String name;

    public String getName() {
        return name;
    }

    private int statusRepeatCount = 0;

    private int exceptionCount = 0;

    static final int ALLOWED_REPEATS = 3;

    public void doAppend(E bag) {
        // WARNING: The guard check MUST be the first statement in the doAppend() method.

        // prevent re-entry.
        if (Boolean.TRUE.equals(guard.get())) {
            return;
        }

        try {
            guard.set(Boolean.TRUE);

            if (!this.started) {
                if (statusRepeatCount++ < ALLOWED_REPEATS) {
                    addStatus("Attempted to append to non started appender [" + name + "].");
                }
                return;
            }

            // ok, we now invoke derived class' implementation of append
            this.append(bag);

        } catch (Exception e) {
            if (exceptionCount++ < ALLOWED_REPEATS) {
                addError("Appender [" + name + "] failed to append.");
            }
        } finally {
            guard.set(Boolean.FALSE);
        }
    }

    abstract protected void append(E bag);

    /**
     * Set the name of this appender.
     */
    public void setName(String name) {
        this.name = name;
    }

    public void start() {
        started = true;
    }

    public void stop() {
        started = false;
    }

    public boolean isStarted() {
        return started;
    }

    public String toString() {
        return this.getClass().getName() + "[" + name + "]";
    }



    protected void addInfo(String msg){
        System.out.println(msg);
    }

    protected void addError(String msg){
        System.out.println(msg);
    }

    protected void addError(String msg,Exception e){
        e.printStackTrace(System.out);
    }

    protected void addStatus(String msg){
        System.out.println(msg);
    }

    protected void addWarn(String msg){
        System.out.println(msg);

    }







}

